<?php
/**
 * 시스템 로그 기록 헬퍼 함수
 * 
 * 사용 예시:
 * writeLog('login', null, null, 1, '사용자 로그인 성공');
 * writeLog('create', null, $jsonData, 1, '가맹점 추가');
 * writeLog('update', $oldData, $newData, 1, '정보 수정');
 */

/**
 * 로그 기록 함수
 * 
 * @param string $type 로그 타입 (login, logout, create, update, delete, error 등)
 * @param string|array|null $oldData 이전 데이터 (배열인 경우 JSON으로 변환)
 * @param string|array|null $newData 변경된 데이터 (배열인 경우 JSON으로 변환)
 * @param int $status 상태 (1: 성공, 0: 실패)
 * @param string $detail 상세 설명
 * @param int|null $userId 사용자 ID (null인 경우 세션에서 가져옴)
 * @return bool 성공 여부
 */
function writeLog($type, $oldData = null, $newData = null, $status = 1, $detail = '', $userId = null) {
    global $con;
    
    // 세션에서 사용자 ID 가져오기
    if ($userId === null && isset($_SESSION['id'])) {
        $userId = $_SESSION['id'];
    }
    
    // 배열인 경우 JSON으로 변환
    if (is_array($oldData)) {
        $oldData = json_encode($oldData, JSON_UNESCAPED_UNICODE);
    }
    
    if (is_array($newData)) {
        $newData = json_encode($newData, JSON_UNESCAPED_UNICODE);
    }
    
    // 데이터베이스 연결 확인
    if (!isset($con) || !$con) {
        error_log("Log Helper: Database connection not available");
        return false;
    }
    
    try {
        // 로그 삽입 쿼리
        $query = "
            INSERT INTO all_log (
                user_id, 
                type, 
                old_data, 
                new_data, 
                status, 
                detail, 
                created_at
            ) VALUES (?, ?, ?, ?, ?, ?, NOW())
        ";
        
        $stmt = mysqli_prepare($con, $query);
        if (!$stmt) {
            error_log("Log Helper: Query prepare failed - " . mysqli_error($con));
            return false;
        }
        
        // NULL 값 처리
        $userId = $userId ?: 0;
        $oldData = $oldData ?: null;
        $newData = $newData ?: null;
        
        mysqli_stmt_bind_param($stmt, "isssiss", 
            $userId,
            $type,
            $oldData,
            $newData,
            $status,
            $detail
        );
        
        $result = mysqli_stmt_execute($stmt);
        
        if (!$result) {
            error_log("Log Helper: Query execution failed - " . mysqli_stmt_error($stmt));
        }
        
        mysqli_stmt_close($stmt);
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Log Helper Error: " . $e->getMessage());
        return false;
    }
}

/**
 * 로그인 로그 기록
 */
function logLogin($userId, $success = true, $detail = '') {
    $status = $success ? 1 : 0;
    $detail = $detail ?: ($success ? '로그인 성공' : '로그인 실패');
    return writeLog('login', null, null, $status, $detail, $userId);
}

/**
 * 로그아웃 로그 기록
 */
function logLogout($userId = null) {
    return writeLog('logout', null, null, 1, '로그아웃', $userId);
}

/**
 * 생성 로그 기록
 */
function logCreate($table, $newData, $detail = '') {
    $detail = $detail ?: "$table 테이블에 데이터 추가";
    return writeLog('create', null, $newData, 1, $detail);
}

/**
 * 수정 로그 기록
 */
function logUpdate($table, $oldData, $newData, $detail = '') {
    $detail = $detail ?: "$table 테이블 데이터 수정";
    return writeLog('update', $oldData, $newData, 1, $detail);
}

/**
 * 삭제 로그 기록
 */
function logDelete($table, $oldData, $detail = '') {
    $detail = $detail ?: "$table 테이블에서 데이터 삭제";
    return writeLog('delete', $oldData, null, 1, $detail);
}

/**
 * 에러 로그 기록
 */
function logError($errorMessage, $context = null) {
    return writeLog('error', null, $context, 0, $errorMessage);
}

/**
 * API 호출 로그 기록
 */
function logApiCall($apiName, $request, $response, $success = true) {
    $status = $success ? 1 : 0;
    $detail = "$apiName API 호출 " . ($success ? '성공' : '실패');
    return writeLog('api', $request, $response, $status, $detail);
}

/**
 * 권한 관련 로그 기록
 */
function logPermission($action, $resource, $allowed = true) {
    $status = $allowed ? 1 : 0;
    $detail = "$resource 에 대한 $action 권한 " . ($allowed ? '허용' : '거부');
    return writeLog('permission', null, null, $status, $detail);
}

// 사용 예시:
/*
// 로그인 성공
logLogin($_POST['user_id'], true, 'IP: ' . $_SERVER['REMOTE_ADDR']);

// 카카오 계정 추가
$newKakaoData = [
    'kakao_tid' => $kakao_tid,
    'client_id' => $client_id,
    'name' => $name
];
logCreate('kakao_account', $newKakaoData, '새 카카오 계정 등록');

// 데이터 수정
$oldData = ['name' => '이전이름', 'status' => 'active'];
$newData = ['name' => '새이름', 'status' => 'inactive'];
logUpdate('member', $oldData, $newData, '회원 정보 수정');

// 에러 발생
logError('Database connection failed', ['server' => $host, 'error' => mysqli_connect_error()]);
*/
?>