<?php
include 'layout/header.php';
?>

<!-- 모바일 메뉴 열기 버튼 -->
<button class="md:hidden fixed top-4 left-4 z-40 bg-primary text-white p-2 rounded-full shadow-lg"
    onclick="toggleSidebar()">
    ☰
</button>

<main class="flex-1 p-8 space-y-8">
    <div class="flex justify-between items-center mb-6">
        <h2 class="text-3xl font-bold text-primary">지급/회수 내역</h2>
        <button onclick="openTransactionModal()" 
            class="px-4 py-2 bg-primary text-white rounded hover:bg-teal-600 transition">
            + 새 거래 추가
        </button>
    </div>

    <!-- 요약 정보 카드 -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-white/5 border border-white/10 rounded-xl p-4">
            <div class="text-gray-400 text-sm mb-1">검색기간 지급</div>
            <div class="text-2xl font-bold text-green-400" id="periodDeposit">0원</div>
            <div class="text-xs text-gray-500 mt-1">오늘: <span id="todayDeposit">0원</span></div>
        </div>
        <div class="bg-white/5 border border-white/10 rounded-xl p-4">
            <div class="text-gray-400 text-sm mb-1">검색기간 회수</div>
            <div class="text-2xl font-bold text-red-400" id="periodWithdraw">0원</div>
            <div class="text-xs text-gray-500 mt-1">오늘: <span id="todayWithdraw">0원</span></div>
        </div>
        <div class="bg-white/5 border border-white/10 rounded-xl p-4">
            <div class="text-gray-400 text-sm mb-1">검색기간 순계</div>
            <div class="text-2xl font-bold text-primary" id="periodNet">0원</div>
            <div class="text-xs text-gray-500 mt-1">오늘: <span id="todayNet">0원</span></div>
        </div>
        <div class="bg-white/5 border border-white/10 rounded-xl p-4">
            <div class="text-gray-400 text-sm mb-1">전체 잔액</div>
            <div class="text-2xl font-bold text-white" id="totalBalance">0원</div>
            <div class="text-xs text-gray-500 mt-1">건수: <span id="totalCount">0건</span></div>
        </div>
    </div>

    <!-- 검색 폼 -->
    <form id="searchForm" class="mb-6 space-y-4">
        <div class="flex flex-wrap gap-4">
            <!-- 날짜 범위 검색 -->
            <div class="flex items-center gap-2">
                <input type="date" name="start_date" id="start_date"
                    class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600">
                <span class="text-gray-400">~</span>
                <input type="date" name="end_date" id="end_date"
                    class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600">
            </div>

            <!-- 회원 ID 검색 -->
            <input type="text" name="member_user_id" id="member_user_id" placeholder="회원 ID 검색"
                class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 w-48 focus:outline-none focus:ring-2 focus:ring-primary">

            <!-- 거래 유형 필터 -->
            <select name="type" id="type"
                class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600">
                <option value="">전체 유형</option>
                <option value="deposit">지급 (입금)</option>
                <option value="withdraw">회수 (출금)</option>
            </select>

            <!-- 상태 필터 -->
            <select name="status" id="status"
                class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600">
                <option value="">전체 상태</option>
                <option value="pending">대기중</option>
                <option value="completed">완료</option>
                <option value="cancelled">취소</option>
            </select>

            <!-- 금액 범위 -->
            <div class="flex items-center gap-2">
                <input type="number" name="min_amount" id="min_amount" placeholder="최소 금액"
                    class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 w-32">
                <span class="text-gray-400">~</span>
                <input type="number" name="max_amount" id="max_amount" placeholder="최대 금액"
                    class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 w-32">
            </div>

            <!-- 검색 버튼 -->
            <button type="submit" 
                class="px-4 py-2 bg-primary text-white rounded hover:bg-teal-600 transition">
                검색
            </button>
            
            <!-- 초기화 버튼 -->
            <button type="button" onclick="resetForm()"
                class="px-4 py-2 bg-gray-600 text-white rounded hover:bg-gray-700 transition">
                초기화
            </button>

            <!-- 엑셀 다운로드 -->
            <!-- <button type="button" onclick="downloadExcel()"
                class="px-4 py-2 bg-green-600 text-white rounded hover:bg-green-700 transition">
                📥 엑셀 다운로드
            </button> -->
        </div>
    </form>

    <!-- 거래 내역 테이블 -->
    <div class="bg-white/5 border border-white/10 rounded-xl overflow-hidden shadow">
        <div class="overflow-x-auto">
            <table class="min-w-full text-sm text-left">
                <thead class="bg-gray-800 text-gray-300">
                    <tr>
                        <th class="px-4 py-3 border-b border-white/10">
                            <input type="checkbox" id="selectAll" class="rounded">
                        </th>
                        <th class="px-4 py-3 border-b border-white/10">ID</th>
                        <th class="px-4 py-3 border-b border-white/10">회원 ID</th>
                        <th class="px-4 py-3 border-b border-white/10">관리자</th>
                        <th class="px-4 py-3 border-b border-white/10">유형</th>
                        <th class="px-4 py-3 border-b border-white/10 text-right">금액</th>
                        <th class="px-4 py-3 border-b border-white/10 text-right">이전 잔액</th>
                        <th class="px-4 py-3 border-b border-white/10 text-right">이후 잔액</th>
                        <th class="px-4 py-3 border-b border-white/10">메모</th>
                        <th class="px-4 py-3 border-b border-white/10">상태</th>
                        <th class="px-4 py-3 border-b border-white/10">IP 주소</th>
                        <th class="px-4 py-3 border-b border-white/10">생성일시</th>
                    </tr>
                </thead>
                <tbody id="transactionTableBody" class="divide-y divide-white/10">
                    <!-- 데이터는 JavaScript로 동적 로드 -->
                </tbody>
            </table>
        </div>
        
        <!-- 로딩 표시 -->
        <div id="loadingIndicator" class="hidden p-8 text-center text-gray-400">
            <div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
            <p class="mt-2">데이터를 불러오는 중...</p>
        </div>
    </div>

    <!-- 페이지네이션 -->
    <div id="pagination" class="flex justify-center space-x-2 text-sm mt-4">
        <!-- 페이지네이션은 JavaScript로 동적 생성 -->
    </div>
</main>

<!-- 거래 추가/수정 모달 -->
<div id="transactionModal" class="fixed inset-0 bg-black/50 hidden items-center justify-center z-50">
    <div class="bg-gray-900 rounded-xl p-6 w-full max-w-md border border-white/10">
        <h3 class="text-xl font-bold text-primary mb-4" id="modalTitle">새 거래 추가</h3>
        
        <form id="transactionForm" class="space-y-4">
            <input type="hidden" id="transactionId" name="id">
            
            <div>
                <label class="block text-sm text-gray-400 mb-1">가맹점 ID <span class="text-red-500">*</span></label>
                <input type="text" id="modalMemberUserId" name="user_id" required
                    class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                    placeholder="가맹점 ID를 입력하세요">
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-1">거래 유형 <span class="text-red-500">*</span></label>
                <select id="modalType" name="type" required
                    class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
                    <option value="">선택하세요</option>
                    <option value="deposit">지급 (가맹점 잔액 증가)</option>
                    <option value="withdraw">회수 (가맹점 잔액 감소)</option>
                </select>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-1">금액 <span class="text-red-500">*</span></label>
                <input type="text" id="modalAmount" name="amount" required
                    class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                    placeholder="0"
                    onkeyup="formatAmountInput(this)"
                    onkeypress="return onlyNumberKey(event)">
                <small class="text-gray-500">숫자만 입력 가능합니다</small>
            </div>
            
            <div>
                <label class="block text-sm text-gray-400 mb-1">메모 <span class="text-red-500">*</span></label>
                <textarea id="modalMemo" name="memo" rows="3" required
                    class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                    placeholder="거래 사유를 입력하세요 (필수)"></textarea>
            </div>
            
            <div class="flex gap-2 justify-end">
                <button type="button" onclick="closeTransactionModal()"
                    class="px-4 py-2 bg-gray-600 text-white rounded hover:bg-gray-700 transition">
                    취소
                </button>
                <button type="submit" id="submitBtn"
                    class="px-4 py-2 bg-primary text-white rounded hover:bg-teal-600 transition">
                    저장
                </button>
            </div>
        </form>
    </div>
</div>

<style>
/* 커스텀 스타일 */
.transaction-row-deposit {
    background: linear-gradient(90deg, rgba(16, 185, 129, 0.05) 0%, transparent 100%);
}

.transaction-row-withdraw {
    background: linear-gradient(90deg, rgba(239, 68, 68, 0.05) 0%, transparent 100%);
}

/* 애니메이션 */
@keyframes slideIn {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.animate-slideIn {
    animation: slideIn 0.3s ease-out;
}

/* 스크롤바 스타일 */
::-webkit-scrollbar {
    width: 8px;
    height: 8px;
}

::-webkit-scrollbar-track {
    background: #1f2937;
}

::-webkit-scrollbar-thumb {
    background: #4b5563;
    border-radius: 4px;
}

::-webkit-scrollbar-thumb:hover {
    background: #6b7280;
}

/* 체크박스 스타일 */
input[type="checkbox"] {
    width: 16px;
    height: 16px;
    cursor: pointer;
}

/* 토스트 메시지 스타일 */
.toast {
    position: fixed;
    bottom: 20px;
    right: 20px;
    padding: 12px 20px;
    border-radius: 8px;
    color: white;
    font-weight: 500;
    z-index: 9999;
    animation: slideIn 0.3s ease-out;
    max-width: 400px;
    word-wrap: break-word;
}

.toast-success {
    background-color: #10b981;
}

.toast-error {
    background-color: #ef4444;
}

.toast-info {
    background-color: #3b82f6;
}

/* 포커스 스타일 개선 */
input:focus, select:focus, textarea:focus {
    outline: none;
    border-color: #14b8a6 !important;
    box-shadow: 0 0 0 3px rgba(20, 184, 166, 0.1);
}

/* 필수 입력 표시 */
.text-red-500 {
    color: #ef4444;
}

/* 버튼 비활성화 스타일 */
button:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

@keyframes slideOut {
    from {
        opacity: 1;
        transform: translateY(0);
    }
    to {
        opacity: 0;
        transform: translateY(20px);
    }
}
</style>

<script>
// 전역 변수
let currentPage = 1;
let isLoading = false;
let selectedRows = new Set();

// HTML 이스케이프 함수
function escapeHtml(text) {
    if (text == null) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return String(text).replace(/[&<>"']/g, m => map[m]);
}

// 날짜 포맷 함수
function formatDate(date) {
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
}

// 날짜시간 포맷 함수
function formatDateTime(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    return date.toLocaleString('ko-KR', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
    });
}

// 숫자 포맷 함수
function formatNumber(num) {
    if (num == null || isNaN(num)) return '0';
    return Number(num).toLocaleString('ko-KR');
}

// 숫자만 입력 가능하도록 처리
function onlyNumberKey(evt) {
    var ASCIICode = (evt.which) ? evt.which : evt.keyCode;
    if (ASCIICode > 31 && (ASCIICode < 48 || ASCIICode > 57))
        return false;
    return true;
}

// 금액 입력 포맷팅 (콤마 추가)
function formatAmountInput(input) {
    // 콤마 제거
    let value = input.value.replace(/[^\d]/g, '');
    
    // 콤마 추가
    if (value) {
        value = parseInt(value).toLocaleString('ko-KR');
    }
    
    input.value = value;
}

// 콤마 제거하여 순수 숫자만 반환
function removeComma(str) {
    return str.replace(/[^\d]/g, '');
}

// 기본 날짜 가져오기
function getDefaultDates() {
    const today = new Date();
    const sevenDaysAgo = new Date(today);
    sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
    
    return {
        startDate: formatDate(sevenDaysAgo),
        endDate: formatDate(today)
    };
}

// 폼 초기화
function resetForm() {
    const defaults = getDefaultDates();
    document.getElementById('searchForm').reset();
    document.getElementById('start_date').value = defaults.startDate;
    document.getElementById('end_date').value = defaults.endDate;
    currentPage = 1;
    fetchTransactions();
}

// URL 파라미터 가져오기
function getUrlParams() {
    const params = new URLSearchParams(window.location.search);
    return {
        page: parseInt(params.get('page')) || 1,
        member_user_id: params.get('member_user_id') || '',
        type: params.get('type') || '',
        status: params.get('status') || '',
        start_date: params.get('start_date') || '',
        end_date: params.get('end_date') || '',
        min_amount: params.get('min_amount') || '',
        max_amount: params.get('max_amount') || ''
    };
}

// 폼 값 설정
function setFormValues() {
    const params = getUrlParams();
    const defaults = getDefaultDates();
    
    document.getElementById('member_user_id').value = params.member_user_id;
    document.getElementById('type').value = params.type;
    document.getElementById('status').value = params.status;
    document.getElementById('start_date').value = params.start_date || defaults.startDate;
    document.getElementById('end_date').value = params.end_date || defaults.endDate;
    document.getElementById('min_amount').value = params.min_amount;
    document.getElementById('max_amount').value = params.max_amount;
    
    currentPage = params.page;
}

// 요약 정보 로드
function loadSummary(data) {
    if (data && data.summary) {
        document.getElementById('periodDeposit').textContent = formatNumber(data.summary.periodDeposit) + '원';
        document.getElementById('periodWithdraw').textContent = formatNumber(data.summary.periodWithdraw) + '원';
        document.getElementById('periodNet').textContent = formatNumber(data.summary.periodNet) + '원';
        document.getElementById('todayDeposit').textContent = formatNumber(data.summary.todayDeposit) + '원';
        document.getElementById('todayWithdraw').textContent = formatNumber(data.summary.todayWithdraw) + '원';
        document.getElementById('todayNet').textContent = formatNumber(data.summary.todayNet) + '원';
        document.getElementById('totalBalance').textContent = formatNumber(data.summary.totalBalance) + '원';
        document.getElementById('totalCount').textContent = 
            `지급 ${data.summary.depositCount}건, 회수 ${data.summary.withdrawCount}건`;
    }
}

// 거래 데이터 가져오기
function fetchTransactions() {
    if (isLoading) return;
    
    isLoading = true;
    const tbody = document.getElementById('transactionTableBody');
    const loadingIndicator = document.getElementById('loadingIndicator');
    
    tbody.innerHTML = '';
    loadingIndicator.classList.remove('hidden');
    
    const formData = new FormData(document.getElementById('searchForm'));
    const params = new URLSearchParams();
    
    for (let [key, value] of formData.entries()) {
        if (value) params.append(key, value);
    }
    
    params.append('page', currentPage);
    params.append('per_page', 20);
    
    fetch(`/ajax/fetch_manual_transactions.php?${params.toString()}`)
        .then(response => response.json())
        .then(data => {
            loadingIndicator.classList.add('hidden');
            
            if (!data.success) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="12" class="px-4 py-8 text-center text-red-400">
                            ${escapeHtml(data.message || '데이터 로드 실패')}
                        </td>
                    </tr>
                `;
                return;
            }
            
            if (!data.transactions || data.transactions.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="12" class="px-4 py-8 text-center text-gray-400">
                            검색 결과가 없습니다.
                        </td>
                    </tr>
                `;
                return;
            }
            
            data.transactions.forEach(transaction => {
                const row = createTransactionRow(transaction);
                tbody.insertAdjacentHTML('beforeend', row);
            });
            
            loadSummary(data);
            updatePagination(data.totalPages, data.currentPage);
            updateUrl();
        })
        .catch(error => {
            console.error('Error:', error);
            loadingIndicator.classList.add('hidden');
            tbody.innerHTML = `
                <tr>
                    <td colspan="12" class="px-4 py-8 text-center text-red-400">
                        오류가 발생했습니다: ${escapeHtml(error.message)}
                    </td>
                </tr>
            `;
        })
        .finally(() => {
            isLoading = false;
        });
}

// 거래 행 생성
function createTransactionRow(transaction) {
    const isDeposit = transaction.type === 'deposit';
    const typeClass = isDeposit ? 'text-green-400' : 'text-red-400';
    const typeText = isDeposit ? '지급 ↓' : '회수 ↑';
    const rowClass = isDeposit ? 'transaction-row-deposit' : 'transaction-row-withdraw';
    
    let statusClass = '';
    let statusText = '';
    
    switch (transaction.status) {
        case 'pending':
            statusClass = 'text-yellow-400';
            statusText = '대기중';
            break;
        case 'completed':
            statusClass = 'text-green-400';
            statusText = '완료';
            break;
        case 'cancelled':
            statusClass = 'text-red-400';
            statusText = '취소';
            break;
        default:
            statusClass = 'text-gray-400';
            statusText = transaction.status || '-';
    }
    
    return `
        <tr class="hover:bg-white/5 transition ${rowClass} animate-slideIn">
            <td class="px-4 py-2">
                <input type="checkbox" value="${transaction.id}" onchange="toggleRowSelection(${transaction.id})"
                    ${selectedRows.has(transaction.id) ? 'checked' : ''}>
            </td>
            <td class="px-4 py-2">${escapeHtml(transaction.id)}</td>
            <td class="px-4 py-2 font-medium text-white">${escapeHtml(transaction.member_user_id || '-')}</td>
            <td class="px-4 py-2">${escapeHtml(transaction.admin_id || 'admin')}</td>
            <td class="px-4 py-2">
                <span class="${typeClass} font-medium">${typeText}</span>
            </td>
            <td class="px-4 py-2 text-right font-medium ${typeClass}">
                ${isDeposit ? '+' : '-'}${formatNumber(transaction.amount)}원
            </td>
            <td class="px-4 py-2 text-right">${formatNumber(transaction.balance_before)}원</td>
            <td class="px-4 py-2 text-right font-medium">${formatNumber(transaction.balance_after)}원</td>
            <td class="px-4 py-2">
                <span class="text-gray-300 text-xs" title="${escapeHtml(transaction.memo || '')}">
                    ${transaction.memo ? escapeHtml(transaction.memo.substring(0, 30)) + (transaction.memo.length > 30 ? '...' : '') : '-'}
                </span>
            </td>
            <td class="px-4 py-2">
                <span class="${statusClass} font-medium">${statusText}</span>
            </td>
            <td class="px-4 py-2 text-gray-400 text-xs">${escapeHtml(transaction.ip_address || '-')}</td>
            <td class="px-4 py-2 text-gray-400 text-xs">${formatDateTime(transaction.created_at)}</td>
        </tr>
    `;
}

// 페이지네이션 업데이트
function updatePagination(totalPages, currentPageNum) {
    const pagination = document.getElementById('pagination');
    pagination.innerHTML = '';
    
    if (totalPages <= 1) return;
    
    if (currentPageNum > 1) {
        pagination.innerHTML += `
            <button onclick="changePage(${currentPageNum - 1})" 
                class="px-3 py-1 bg-gray-800 hover:bg-gray-700 rounded transition">
                이전
            </button>
        `;
    }
    
    let startPage = Math.max(1, currentPageNum - 2);
    let endPage = Math.min(totalPages, startPage + 4);
    
    if (startPage > 1) {
        pagination.innerHTML += `
            <button onclick="changePage(1)" 
                class="px-3 py-1 bg-gray-800 hover:bg-gray-700 rounded transition">
                1
            </button>
        `;
        if (startPage > 2) {
            pagination.innerHTML += `<span class="px-2 text-gray-400">...</span>`;
        }
    }
    
    for (let i = startPage; i <= endPage; i++) {
        const isActive = i === currentPageNum;
        pagination.innerHTML += `
            <button onclick="changePage(${i})" 
                class="px-3 py-1 ${isActive ? 'bg-primary text-white' : 'bg-gray-800 hover:bg-gray-700'} rounded transition">
                ${i}
            </button>
        `;
    }
    
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            pagination.innerHTML += `<span class="px-2 text-gray-400">...</span>`;
        }
        pagination.innerHTML += `
            <button onclick="changePage(${totalPages})" 
                class="px-3 py-1 bg-gray-800 hover:bg-gray-700 rounded transition">
                ${totalPages}
            </button>
        `;
    }
    
    if (currentPageNum < totalPages) {
        pagination.innerHTML += `
            <button onclick="changePage(${currentPageNum + 1})" 
                class="px-3 py-1 bg-gray-800 hover:bg-gray-700 rounded transition">
                다음
            </button>
        `;
    }
}

// 페이지 변경
function changePage(page) {
    currentPage = page;
    fetchTransactions();
}

// URL 업데이트
function updateUrl() {
    const formData = new FormData(document.getElementById('searchForm'));
    const params = new URLSearchParams();
    
    for (let [key, value] of formData.entries()) {
        if (value) params.append(key, value);
    }
    if (currentPage > 1) params.append('page', currentPage);
    
    const newUrl = window.location.pathname + (params.toString() ? '?' + params.toString() : '');
    window.history.pushState({}, '', newUrl);
}

// 모달 열기
function openTransactionModal(id = null) {
    const modal = document.getElementById('transactionModal');
    const modalTitle = document.getElementById('modalTitle');
    const form = document.getElementById('transactionForm');
    
    form.reset();
    
    if (id) {
        modalTitle.textContent = '거래 수정';
        document.getElementById('transactionId').value = id;
        loadTransactionData(id);
    } else {
        modalTitle.textContent = '새 거래 추가';
        document.getElementById('transactionId').value = '';
    }
    
    modal.classList.remove('hidden');
    modal.classList.add('flex');
}

// 모달 닫기
function closeTransactionModal() {
    const modal = document.getElementById('transactionModal');
    modal.classList.add('hidden');
    modal.classList.remove('flex');
    document.getElementById('transactionForm').reset();
}

// 토스트 메시지 표시 함수 (단일 함수로 통합)
function showToast(message, type = 'info') {
    // 기존 토스트가 있으면 제거
    const existingToast = document.querySelector('.toast');
    if (existingToast) {
        existingToast.remove();
    }
    
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.textContent = message;
    document.body.appendChild(toast);
    
    // 3초 후 자동 제거
    setTimeout(() => {
        toast.style.animation = 'slideOut 0.3s ease-out';
        setTimeout(() => {
            toast.remove();
        }, 300);
    }, 3000);
}

// 거래 데이터 로드 (수정용)
function loadTransactionData(id) {
    fetch(`/ajax/get_transaction.php?id=${id}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('modalMemberUserId').value = data.transaction.member_user_id;
                document.getElementById('modalType').value = data.transaction.type;
                document.getElementById('modalAmount').value = formatNumber(data.transaction.amount);
                document.getElementById('modalMemo').value = data.transaction.memo || '';
            }
        })
        .catch(error => console.error('Error loading transaction:', error));
}

// 행 선택 토글
function toggleRowSelection(id) {
    if (selectedRows.has(id)) {
        selectedRows.delete(id);
    } else {
        selectedRows.add(id);
    }
}

// 엑셀 다운로드
function downloadExcel() {
    const formData = new FormData(document.getElementById('searchForm'));
    const params = new URLSearchParams();
    
    for (let [key, value] of formData.entries()) {
        if (value) params.append(key, value);
    }
    
    window.location.href = `/ajax/export_transactions.php?${params.toString()}`;
}

// 모바일 사이드바 토글
function toggleSidebar() {
    const sidebar = document.querySelector('.sidebar');
    if (sidebar) {
        sidebar.classList.toggle('hidden');
    }
}

// 검색 폼 제출
document.getElementById('searchForm').addEventListener('submit', function(e) {
    e.preventDefault();
    currentPage = 1;
    fetchTransactions();
});

// 거래 폼 제출 (process_transaction.php 사용) - 단일 이벤트 리스너
document.getElementById('transactionForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const submitBtn = document.getElementById('submitBtn');
    const originalText = submitBtn.textContent;
    
    // 버튼 비활성화 및 로딩 표시
    submitBtn.disabled = true;
    submitBtn.textContent = '처리중...';
    
    // FormData 생성
    const formData = new FormData();
    formData.append('user_id', document.getElementById('modalMemberUserId').value.trim());
    formData.append('type', document.getElementById('modalType').value);
    formData.append('amount', removeComma(document.getElementById('modalAmount').value));
    formData.append('memo', document.getElementById('modalMemo').value.trim());
    
    // AJAX 요청
    fetch('/ajax/process_transaction.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(result => {
        if (result.success) {
            showToast(result.message, 'success');
            closeTransactionModal();
            fetchTransactions();
        } else {
            showToast(result.message || '처리 중 오류가 발생했습니다.', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('서버 오류가 발생했습니다. 잠시 후 다시 시도해주세요.', 'error');
    })
    .finally(() => {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
    });
});

// 전체 선택
document.getElementById('selectAll').addEventListener('change', function() {
    const checkboxes = document.querySelectorAll('#transactionTableBody input[type="checkbox"]');
    checkboxes.forEach(checkbox => {
        checkbox.checked = this.checked;
        const id = parseInt(checkbox.value);
        if (this.checked) {
            selectedRows.add(id);
        } else {
            selectedRows.delete(id);
        }
    });
});

// 페이지 로드 시 초기화
document.addEventListener('DOMContentLoaded', function() {
    setFormValues();
    fetchTransactions();
    
    // 5분마다 자동 새로고침
    setInterval(function() {
        if (!isLoading) {
            fetchTransactions();
        }
    }, 300000);
});

// ESC 키로 모달 닫기
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        const modal = document.getElementById('transactionModal');
        if (!modal.classList.contains('hidden')) {
            closeTransactionModal();
        }
    }
});

// 모달 외부 클릭시 닫기
document.getElementById('transactionModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeTransactionModal();
    }
});
</script>

</body>
</html>