<?php
/**
 * 10분 이상 보류(pending) 상태인 입금 신청을 자동으로 취소 처리
 * all_log 테이블에 처리 내역 기록
 */

// 에러 표시 설정
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 데이터베이스 연결 (writeLog 함수 포함)
require_once __DIR__ . '/../inc/db_connect.php';

// 설정 값
$timeoutMinutes = 10;  // 자동 취소까지 대기 시간 (분)
$batchLimit = 50;      // 한 번에 처리할 최대 건수
$testMode = false;     // 테스트 모드 (true: 실제 업데이트 안함, 대상만 확인)

// 실행 시작 로그
$mode = $testMode ? "[TEST MODE]" : "";
error_log("[AUTO_CANCEL]{$mode} Starting auto cancel process at " . date('Y-m-d H:i:s'));

try {
    // 트랜잭션 시작
    mysqli_begin_transaction($con);
    
    // 1. 10분 이상 경과한 보류 상태 입금 신청 조회
    $selectQuery = "
        SELECT 
            t.id,
            t.user_id,
            t.amount,
            t.status,
            t.created_at,
            m.user_name,
            m.user_id as member_user_id,
            TIMESTAMPDIFF(MINUTE, t.created_at, NOW()) as elapsed_minutes
        FROM transfer t
        JOIN member m ON t.user_id = m.id
        WHERE t.type = 'deposit' 
            AND t.status = 'pending'
            AND t.created_at <= DATE_SUB(NOW(), INTERVAL ? MINUTE)
        ORDER BY t.created_at ASC
        LIMIT ?
    ";
    
    $selectStmt = mysqli_prepare($con, $selectQuery);
    if (!$selectStmt) {
        throw new Exception("Prepare failed: " . mysqli_error($con));
    }
    mysqli_stmt_bind_param($selectStmt, "ii", $timeoutMinutes, $batchLimit);
    mysqli_stmt_execute($selectStmt);
    $result = mysqli_stmt_get_result($selectStmt);
    
    $cancelledCount = 0;
    $cancelledIds = [];
    $cancelledDetails = [];
    $errors = [];
    
    // 2. 각 항목에 대해 취소 처리
    while ($row = mysqli_fetch_assoc($result)) {
        try {
            $transferId = $row['id'];
            $memberId = $row['user_id'];
            $memberUserId = $row['member_user_id'];
            $userName = $row['user_name'];
            $amount = $row['amount'];
            $elapsedMinutes = $row['elapsed_minutes'];
            $createdAt = $row['created_at'];
            
            // 테스트 모드인 경우 실제 업데이트하지 않고 로그만
            if ($testMode) {
                error_log("[AUTO_CANCEL][TEST] Would cancel: ID={$transferId}, User={$userName}, Amount=" . number_format($amount) . "원, Elapsed={$elapsedMinutes}분");
                $cancelledCount++;
                $cancelledIds[] = $transferId;
                continue;
            }
            
            // transfer 테이블 업데이트 (보류 -> 취소)
            $updateQuery = "UPDATE transfer 
                           SET status = 'cancelled', 
                               cancelled_at = NOW(), 
                               updated_at = NOW() 
                           WHERE id = ? 
                               AND status = 'pending'";
            
            $updateStmt = mysqli_prepare($con, $updateQuery);
            mysqli_stmt_bind_param($updateStmt, "i", $transferId);
            
            if (mysqli_stmt_execute($updateStmt)) {
                $affectedRows = mysqli_stmt_affected_rows($updateStmt);
                
                if ($affectedRows > 0) {
                    $cancelledCount++;
                    $cancelledIds[] = $transferId;
                    
                    // 취소 상세 정보 저장
                    $cancelledDetails[] = [
                        'transfer_id' => $transferId,
                        'member_id' => $memberId,
                        'member_user_id' => $memberUserId,
                        'user_name' => $userName,
                        'amount' => $amount,
                        'elapsed_minutes' => $elapsedMinutes,
                        'created_at' => $createdAt
                    ];
                    
                    // all_log 테이블에 기록
                    $oldData = [
                        'transfer_id' => $transferId,
                        'status' => 'pending',
                        'amount' => $amount,
                        'created_at' => $createdAt
                    ];
                    
                    $newData = [
                        'transfer_id' => $transferId,
                        'status' => 'cancelled',
                        'amount' => $amount,
                        'cancelled_at' => date('Y-m-d H:i:s')
                    ];
                    
                    $detail = "입금 신청 자동 취소 - ID: {$transferId}, 사용자: {$userName}, 금액: " . number_format($amount) . "원, 경과시간: {$elapsedMinutes}분";
                    
                    // writeLog 함수 사용 (db_connect.php에 정의된 함수)
                    writeLog($con, 'AUTO_CANCEL_DEPOSIT', $oldData, $newData, 1, $detail, 0);
                    
                    // transfer_status_logs 테이블에도 기록 (선택사항)
                    try {
                        $logQuery = "INSERT INTO transfer_status_logs 
                                    (transfer_id, old_status, new_status, created_at) 
                                    VALUES (?, 'pending', 'cancelled', NOW())";
                        $logStmt = mysqli_prepare($con, $logQuery);
                        if ($logStmt) {
                            mysqli_stmt_bind_param($logStmt, "i", $transferId);
                            mysqli_stmt_execute($logStmt);
                            mysqli_stmt_close($logStmt);
                        }
                    } catch (Exception $logEx) {
                        // 로그 실패는 무시 (선택사항)
                        error_log("[AUTO_CANCEL] Status log insert failed for ID {$transferId}: " . $logEx->getMessage());
                    }
                    
                    // 개별 취소 성공 로그
                    error_log("[AUTO_CANCEL] Cancelled: ID={$transferId}, User={$userName}, Amount=" . number_format($amount) . "원, Elapsed={$elapsedMinutes}분");
                }
            } else {
                throw new Exception("Failed to update transfer ID {$transferId}: " . mysqli_error($con));
            }
            
            mysqli_stmt_close($updateStmt);
            
        } catch (Exception $e) {
            $errorMsg = "Transfer ID {$transferId}: " . $e->getMessage();
            $errors[] = $errorMsg;
            error_log("[AUTO_CANCEL] Error processing ID {$transferId}: " . $e->getMessage());
            
            // 실패 로그 기록
            $errorDetail = "입금 신청 자동 취소 실패 - ID: {$transferId}, 오류: " . $e->getMessage();
            writeLog($con, 'AUTO_CANCEL_DEPOSIT_ERROR', 
                    ['transfer_id' => $transferId], 
                    null, 
                    0, 
                    $errorDetail, 
                    0);
        }
    }
    
    mysqli_stmt_close($selectStmt);
    
    // 3. 트랜잭션 커밋 또는 롤백
    if ($testMode) {
        mysqli_rollback($con);
        error_log("[AUTO_CANCEL][TEST] Test mode - transaction rolled back");
    } else {
        mysqli_commit($con);
    }
    
    // 4. 실행 결과 로그
    if ($cancelledCount > 0) {
        $idList = implode(', ', $cancelledIds);
        error_log("[AUTO_CANCEL]{$mode} Successfully cancelled {$cancelledCount} pending deposits. IDs: {$idList}");
        
        // 전체 처리 성공 로그 (all_log)
        $summaryDetail = "입금 자동 취소 배치 완료 - 처리 건수: {$cancelledCount}건, ID: [{$idList}]";
        writeLog($con, 'AUTO_CANCEL_BATCH', 
                null, 
                ['cancelled_count' => $cancelledCount, 'cancelled_ids' => $cancelledIds], 
                1, 
                $summaryDetail, 
                0);
    } else {
        error_log("[AUTO_CANCEL]{$mode} No pending deposits to cancel (all within {$timeoutMinutes} minutes)");
        // 처리 대상 없음인 경우 all_log에 기록하지 않음
    }
    
    if (!empty($errors)) {
        error_log("[AUTO_CANCEL] Errors occurred: " . implode('; ', $errors));
    }
    
    // 헤더 설정 (JSON 응답)
    header('Content-Type: application/json; charset=UTF-8');
    
    // 성공 응답
    echo json_encode([
        'success' => true,
        'test_mode' => $testMode,
        'cancelled_count' => $cancelledCount,
        'cancelled_ids' => $cancelledIds,
        'cancelled_details' => $cancelledDetails,
        'errors' => $errors,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // 트랜잭션 롤백
    mysqli_rollback($con);
    
    // 에러 로그
    error_log("[AUTO_CANCEL] Fatal error: " . $e->getMessage());
    
    // 심각한 오류 로그 (all_log)
    writeLog($con, 'AUTO_CANCEL_FATAL_ERROR', 
            null, 
            ['error' => $e->getMessage()], 
            0, 
            "입금 자동 취소 배치 실행 중 심각한 오류 발생: " . $e->getMessage(), 
            0);
    
    // 헤더 설정 (JSON 응답)
    header('Content-Type: application/json; charset=UTF-8');
    
    // 에러 응답
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);
    
} finally {
    // 데이터베이스 연결 종료
    if (isset($con) && $con) {
        mysqli_close($con);
    }
    
    // 실행 종료 로그
    $mode = $testMode ? "[TEST MODE]" : "";
    error_log("[AUTO_CANCEL]{$mode} Process completed at " . date('Y-m-d H:i:s'));
}
?>