<?php
include 'layout/header.php';

// 파트너 권한 체크 - 관리자가 아닌 경우만 허용
if ($row['grade'] == 'admin') {
    header('Location: index.php');
    exit;
}

// 카카오 계정 목록 가져오기
$kakao_accounts = [];
if (isset($con)) {
  $kakao_query = "SELECT id, name, kakao_tid FROM kakao_account ORDER BY name";
  $kakao_result = mysqli_query($con, $kakao_query);

  if ($kakao_result) {
    while ($row_kakao = mysqli_fetch_assoc($kakao_result)) {
      $kakao_accounts[] = $row_kakao;
    }
    mysqli_free_result($kakao_result);
  } else {
    error_log("카카오 계정 조회 오류: " . mysqli_error($con));
  }
}

// 은행 계정 목록 가져오기
$bank_accounts = [];
if (isset($con)) {
  $bank_query = "SELECT id, bank, account_holder, account_number FROM bank_account ORDER BY bank, account_holder";
  $bank_result = mysqli_query($con, $bank_query);

  if ($bank_result) {
    while ($row_bank = mysqli_fetch_assoc($bank_result)) {
      $bank_accounts[] = $row_bank;
    }
    mysqli_free_result($bank_result);
  } else {
    error_log("은행 계정 조회 오류: " . mysqli_error($con));
  }
}
?>

<style>
/* 트리뷰 테이블 스타일 */
.tree-table {
  width: 100%;
  border-collapse: collapse;
}

.tree-table th {
  background-color: rgb(31, 41, 55);
  color: rgb(209, 213, 219);
  padding: 12px;
  text-align: left;
  border-bottom: 1px solid rgba(255, 255, 255, 0.1);
  position: sticky;
  top: 0;
  z-index: 10;
}

.tree-table td {
  padding: 8px 12px;
  border-bottom: 1px solid rgba(255, 255, 255, 0.05);
  vertical-align: middle;
}

.tree-row:hover {
  background-color: rgba(255, 255, 255, 0.05);
}

.tree-toggle {
  width: 20px;
  height: 20px;
  display: inline-flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  user-select: none;
  transition: transform 0.2s;
  margin-right: 8px;
}

.tree-toggle.collapsed {
  transform: rotate(-90deg);
}

.tree-indent {
  display: inline-block;
}

.tree-children.hidden {
  display: none;
}

/* 레벨별 색상 - 파트너 시점으로 조정 */
.level-badge {
  display: inline-flex;
  align-items: center;
  padding: 0.125rem 0.5rem;
  border-radius: 9999px;
  font-size: 0.75rem;
  font-weight: 500;
  white-space: nowrap;
}

.level-1 { background-color: rgb(37, 99, 235); color: white; }
.level-2 { background-color: rgb(34, 197, 94); color: white; }
.level-3 { background-color: rgb(250, 204, 21); color: black; }
.level-4 { background-color: rgb(251, 146, 60); color: white; }
.level-5 { background-color: rgb(239, 68, 68); color: white; }

/* 파트너 본인 표시 */
.partner-self {
  background-color: rgba(139, 92, 246, 0.2);
  border-left: 4px solid #8b5cf6;
}

/* 드롭다운 관련 스타일 */
.dropdown-container {
  position: relative;
  display: inline-block;
}

.dropdown-menu {
  position: absolute !important;
  z-index: 9999 !important;
  min-width: 200px;
  top: 100%;
  left: 0;
  margin-top: 5px;
}
</style>

<main class="flex-1 p-8 space-y-8">
  <div class="flex justify-between items-center">
    <div>
      <h2 class="text-3xl font-bold text-primary mb-2">내 하부 가맹점</h2>
      <p class="text-gray-400">나의 하부 가맹점들을 관리할 수 있습니다.</p>
    </div>
    <div class="bg-blue-600/20 px-4 py-2 rounded-lg border border-blue-600/30">
      <span class="text-blue-400 font-semibold"><?= htmlspecialchars($row['user_name']) ?></span>
      <span class="text-gray-400 ml-2">파트너</span>
    </div>
  </div>

  <!-- 하부 가맹점 추가 폼 -->
  <!-- <div class="bg-white/5 border border-white/10 rounded-xl overflow-hidden shadow mb-6">
    <div class="p-4">
      <h3 class="text-lg font-semibold text-white mb-4">하부 가맹점 추가</h3>
      <form id="addSubStoreForm" class="flex flex-wrap gap-4 items-end">
        <input type="hidden" name="parent_user_id" value="<?= htmlspecialchars($row['user_id']) ?>">
        
        <div class="flex-1 min-w-[150px]">
          <label class="block text-sm text-gray-400 mb-1">가맹점 아이디</label>
          <input type="text" name="user_id" required
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
        </div>
        <div class="flex-1 min-w-[200px]">
          <label class="block text-sm text-gray-400 mb-1">가맹점명</label>
          <input type="text" name="user_name" required
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
        </div>
        <div class="flex-1 min-w-[150px]">
          <label class="block text-sm text-gray-400 mb-1">비밀번호</label>
          <input type="password" name="password" required
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
        </div>
        <div class="flex-1 min-w-[120px]">
          <label class="block text-sm text-gray-400 mb-1">가맹점 코드</label>
          <input type="text" name="code" required
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
        </div>
        <div class="flex-1 min-w-[180px]">
          <label class="block text-sm text-gray-400 mb-1">카카오 연결계정</label>
          <select name="kakao_account_id" id="kakao_account_select"
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
            onchange="handleAccountSelection('kakao')">
            <option value="">선택하세요</option>
            <?php foreach ($kakao_accounts as $account): ?>
              <option value="<?= $account['id'] ?>"><?= htmlspecialchars($account['name']) ?>
                (<?= htmlspecialchars($account['kakao_tid']) ?>)</option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="flex-1 min-w-[180px]">
          <label class="block text-sm text-gray-400 mb-1">은행 연결계정</label>
          <select name="bank_account_id" id="bank_account_select"
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
            onchange="handleAccountSelection('bank')">
            <option value="">선택하세요</option>
            <?php foreach ($bank_accounts as $account): ?>
              <option value="<?= $account['id'] ?>"><?= htmlspecialchars($account['bank']) ?> |
                <?= htmlspecialchars($account['account_holder']) ?> (<?= htmlspecialchars($account['account_number']) ?>)
              </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="flex-1 min-w-[150px]">
          <label class="block text-sm text-gray-400 mb-1">개별 출금은행 <span class="text-red-400">*</span></label>
          <select name="bank" required
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            <option value="">선택하세요</option>
            <option value="KB국민은행">KB국민은행</option>
            <option value="신한은행">신한은행</option>
            <option value="우리은행">우리은행</option>
            <option value="하나은행">하나은행</option>
            <option value="NH농협은행">NH농협은행</option>
            <option value="IBK기업은행">IBK기업은행</option>
            <option value="카카오뱅크">카카오뱅크</option>
            <option value="토스뱅크">토스뱅크</option>
            <option value="케이뱅크">케이뱅크</option>
          </select>
        </div>
        <div class="flex-1 min-w-[200px]">
          <label class="block text-sm text-gray-400 mb-1">개별 출금계좌 <span class="text-red-400">*</span></label>
          <input type="text" name="account_number" required placeholder="계좌번호를 입력하세요"
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
        </div>
        <div class="flex-1 min-w-[150px]">
          <label class="block text-sm text-gray-400 mb-1">개별 예금주 <span class="text-red-400">*</span></label>
          <input type="text" name="account_holder" required placeholder="예금주를 입력하세요"
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
        </div>
        <div class="flex-1 min-w-[120px]">
          <label class="block text-sm text-gray-400 mb-1">수수료율 (%)</label>
          <input type="number" name="rate" step="0.01" min="<?= $row['rate'] ?>" max="100" value="<?= $row['rate'] ?>"
            class="w-full px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
          <p class="text-xs text-yellow-400 mt-1">최소: <?= $row['rate'] ?>%</p>
        </div>
        <button type="submit" class="px-6 py-2 bg-primary text-white rounded hover:bg-teal-600 transition">
          추가
        </button>
      </form>

      <div class="mt-3 text-sm text-gray-400">
        <span class="text-yellow-400">ℹ</span> 하부 가맹점의 수수료율은 본인의 수수료율(<?= $row['rate'] ?>%)보다 높거나 같아야 합니다.
      </div>
    </div>
  </div> -->

  <!-- 검색 필터 및 트리 컨트롤 -->
  <div class="flex justify-between items-center mb-6">
    <form method="GET" class="flex gap-4 flex-wrap">
      <input type="text" id="searchInput" placeholder="가맹점명, 아이디, 코드 검색" 
        value=""
        class="px-4 py-2 rounded bg-gray-800 text-white border border-gray-600 w-72 focus:outline-none focus:ring-2 focus:ring-primary">
      
      <button type="button" onclick="searchStores()" class="px-4 py-2 bg-primary text-white rounded hover:bg-teal-600">
        검색
      </button>
    </form>

    <!-- 트리 컨트롤 버튼 -->
    <div class="flex gap-2">
      <button onclick="expandAll()" class="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700">
        모두 펼치기
      </button>
      <button onclick="collapseAll()" class="px-4 py-2 bg-gray-600 text-white rounded hover:bg-gray-700">
        모두 닫기
      </button>
    </div>
  </div>

  <!-- 가맹점 트리뷰 테이블 -->
  <div class="bg-white/5 border border-white/10 rounded-xl shadow overflow-x-auto" style="height:1200px">
    <table class="tree-table text-sm">
      <thead>
        <tr>
          <th width="5%">ID</th>
          <th width="8%">단계</th>
          <th width="10%">가맹점 아이디</th>
          <th width="12%">가맹점명</th>
          <th width="8%">가맹점 코드</th>
          <th width="10%">카카오 연결계정</th>
          <th width="12%">은행 연결계정</th>
          <th width="7%">수수료율</th>
          <th width="10%">잔액</th>
          <th width="8%">등록일</th>
          <th width="10%">관리</th>
        </tr>
      </thead>
      <tbody id="treeTableBody">
        <!-- 트리 행이 여기에 동적으로 생성됩니다 -->
      </tbody>
    </table>
  </div>

  <!-- 하부 가맹점 추가 모달 -->
  <div id="subMemberModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-gray-900 border border-gray-700 rounded-xl p-6 w-full max-w-4xl max-h-[90vh] overflow-y-auto">
      <!-- 모달 헤더 -->
      <div class="flex justify-between items-center mb-6">
        <div>
          <h2 class="text-2xl font-bold text-white">하부 가맹점 추가</h2>
          <p class="text-sm text-gray-400 mt-1">
            상위 가맹점: <span id="parentUserName" class="text-primary font-semibold"></span>
          </p>
        </div>
        <button onclick="closeSubMemberModal()" class="text-gray-400 hover:text-white transition">
          <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
          </svg>
        </button>
      </div>
      
      <input type="hidden" id="parentUserId" value="">
      <input type="hidden" id="parentRate" value="">
      
      <!-- 하부 가맹점 추가 폼 -->
      <form id="subMemberForm" class="space-y-4">
        <!-- 기본 정보 섹션 -->
        <div class="bg-gray-800 rounded-lg p-4 space-y-4">
          <h3 class="text-lg font-semibold text-white mb-3">기본 정보</h3>
          
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm text-gray-400 mb-2">가맹점 아이디 <span class="text-red-400">*</span></label>
              <input type="text" name="user_id" required
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">가맹점명 <span class="text-red-400">*</span></label>
              <input type="text" name="user_name" required
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">비밀번호 <span class="text-red-400">*</span></label>
              <input type="password" name="password" required
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">가맹점 코드 <span class="text-red-400">*</span></label>
              <input type="text" name="code" required
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
          </div>
        </div>
        
        <!-- 연결 계정 섹션 -->
        <div class="bg-gray-800 rounded-lg p-4 space-y-4">
          <h3 class="text-lg font-semibold text-white mb-3">연결 계정 설정</h3>
          <p class="text-sm text-yellow-400 mb-3">※ 카카오 연결계정과 은행 연결계정 중 하나만 선택 가능합니다.</p>
          
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm text-gray-400 mb-2">카카오 연결계정</label>
              <select name="kakao_account_id" id="sub_kakao_account_select"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                onchange="handleSubAccountSelection('kakao')">
                <option value="">선택하세요</option>
                <?php foreach ($kakao_accounts as $account): ?>
                  <option value="<?= $account['id'] ?>"><?= htmlspecialchars($account['name']) ?>
                    (<?= htmlspecialchars($account['kakao_tid']) ?>)</option>
                <?php endforeach; ?>
              </select>
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">은행 연결계정</label>
              <select name="bank_account_id" id="sub_bank_account_select"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                onchange="handleSubAccountSelection('bank')">
                <option value="">선택하세요</option>
                <?php foreach ($bank_accounts as $account): ?>
                  <option value="<?= $account['id'] ?>"><?= htmlspecialchars($account['bank']) ?> |
                    <?= htmlspecialchars($account['account_holder']) ?> (<?= htmlspecialchars($account['account_number']) ?>)
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
        </div>
        
        <!-- 개별 출금 정보 섹션 -->
        <div class="bg-gray-800 rounded-lg p-4 space-y-4">
          <h3 class="text-lg font-semibold text-white mb-3">개별 출금 정보</h3>
          
          <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <label class="block text-sm text-gray-400 mb-2">출금 은행 <span class="text-red-400">*</span></label>
              <select name="bank" required
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
                <option value="">선택하세요</option>
                <option value="KB국민은행">KB국민은행</option>
                <option value="신한은행">신한은행</option>
                <option value="우리은행">우리은행</option>
                <option value="하나은행">하나은행</option>
                <option value="NH농협은행">NH농협은행</option>
                <option value="IBK기업은행">IBK기업은행</option>
                <option value="카카오뱅크">카카오뱅크</option>
                <option value="토스뱅크">토스뱅크</option>
                <option value="케이뱅크">케이뱅크</option>
              </select>
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">계좌번호 <span class="text-red-400">*</span></label>
              <input type="text" name="account_number" required placeholder="계좌번호를 입력하세요"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">예금주 <span class="text-red-400">*</span></label>
              <input type="text" name="account_holder" required placeholder="예금주를 입력하세요"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
          </div>
        </div>
        
        <!-- 수수료 설정 섹션 -->
        <div class="bg-gray-800 rounded-lg p-4">
          <h3 class="text-lg font-semibold text-white mb-3">수수료 설정</h3>
          
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm text-gray-400 mb-2">수수료율 (%)</label>
              <input type="number" name="rate" step="0.01" min="0" max="100" value="0" id="subRateInput"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
              <p class="text-xs text-yellow-400 mt-1" id="rateHelpText">상위 가맹점 수수료율을 확인 중...</p>
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">초기 잔액</label>
              <input type="number" name="initial_balance" step="1" min="0" value="0"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
              <p class="text-xs text-gray-500 mt-1">선택사항입니다.</p>
            </div>
          </div>
        </div>
        
        <!-- 버튼 영역 -->
        <div class="flex justify-end space-x-3 pt-4 border-t border-gray-700">
          <button type="button" onclick="closeSubMemberModal()"
            class="px-6 py-2 bg-gray-700 text-white rounded hover:bg-gray-600 transition">
            취소
          </button>
          <button type="submit"
            class="px-6 py-2 bg-primary text-white rounded hover:bg-teal-600 transition">
            하부 가맹점 추가
          </button>
        </div>
      </form>
    </div>
  </div>

  <!-- 가맹점 수정 모달 -->
  <div id="editMemberModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-gray-900 border border-gray-700 rounded-xl p-6 w-full max-w-4xl max-h-[90vh] overflow-y-auto">
      <!-- 모달 헤더 -->
      <div class="flex justify-between items-center mb-6">
        <div>
          <h2 class="text-2xl font-bold text-white">가맹점 정보 수정</h2>
          <p class="text-sm text-gray-400 mt-1">
            가맹점 ID: <span id="editUserId" class="text-primary font-semibold"></span>
          </p>
        </div>
        <button onclick="closeEditModal()" class="text-gray-400 hover:text-white transition">
          <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
          </svg>
        </button>
      </div>
      
      <!-- 수정 폼 -->
      <form id="editMemberForm" class="space-y-4">
        <input type="hidden" id="edit_user_id" name="user_id">
        
        <!-- 기본 정보 섹션 -->
        <div class="bg-gray-800 rounded-lg p-4 space-y-4">
          <h3 class="text-lg font-semibold text-white mb-3">기본 정보</h3>
          
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm text-gray-400 mb-2">가맹점명 <span class="text-red-400">*</span></label>
              <input type="text" id="edit_user_name" name="user_name" required maxlength="100"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">가맹점 코드 <span class="text-red-400">*</span></label>
              <input type="text" id="edit_code" name="code" required pattern="[0-9-]+" title="숫자와 하이픈(-)만 입력 가능합니다" placeholder="가맹점 코드를 입력하세요"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
          </div>
        </div>
        
        <!-- 연결 계정 섹션 -->
        <div class="bg-gray-800 rounded-lg p-4 space-y-4">
          <h3 class="text-lg font-semibold text-white mb-3">연결 계정 설정</h3>
          <p class="text-sm text-yellow-400 mb-3">※ 카카오 연결계정과 은행 연결계정 중 하나만 선택 가능합니다.</p>
          
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm text-gray-400 mb-2">카카오 연결계정</label>
              <select name="kakao_account_id" id="edit_kakao_account"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                onchange="handleEditAccountSelection('kakao')">
                <option value="">선택하세요</option>
                <?php foreach ($kakao_accounts as $account): ?>
                  <option value="<?= $account['id'] ?>"><?= htmlspecialchars($account['name']) ?>
                    (<?= htmlspecialchars($account['kakao_tid']) ?>)</option>
                <?php endforeach; ?>
              </select>
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">은행 연결계정</label>
              <select name="bank_account_id" id="edit_bank_account"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary"
                onchange="handleEditAccountSelection('bank')">
                <option value="">선택하세요</option>
                <?php foreach ($bank_accounts as $account): ?>
                  <option value="<?= $account['id'] ?>"><?= htmlspecialchars($account['bank']) ?> |
                    <?= htmlspecialchars($account['account_holder']) ?> (<?= htmlspecialchars($account['account_number']) ?>)
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
        </div>
        
        <!-- 개별 출금 정보 섹션 -->
        <div class="bg-gray-800 rounded-lg p-4 space-y-4">
          <h3 class="text-lg font-semibold text-white mb-3">개별 출금 정보</h3>
          
          <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <label class="block text-sm text-gray-400 mb-2">출금 은행 <span class="text-red-400">*</span></label>
              <select id="edit_bank" name="bank" required
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
                <option value="">선택하세요</option>
                <option value="KB국민은행">KB국민은행</option>
                <option value="신한은행">신한은행</option>
                <option value="우리은행">우리은행</option>
                <option value="하나은행">하나은행</option>
                <option value="NH농협은행">NH농협은행</option>
                <option value="IBK기업은행">IBK기업은행</option>
                <option value="카카오뱅크">카카오뱅크</option>
                <option value="토스뱅크">토스뱅크</option>
                <option value="케이뱅크">케이뱅크</option>
              </select>
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">계좌번호 <span class="text-red-400">*</span></label>
              <input type="text" id="edit_account_number" name="account_number" required placeholder="계좌번호를 입력하세요"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">예금주 <span class="text-red-400">*</span></label>
              <input type="text" id="edit_account_holder" name="account_holder" required placeholder="예금주를 입력하세요"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
            </div>
          </div>
        </div>
        
        <!-- 수수료 설정 섹션 -->
        <div class="bg-gray-800 rounded-lg p-4">
          <h3 class="text-lg font-semibold text-white mb-3">수수료 설정</h3>
          
          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label class="block text-sm text-gray-400 mb-2">수수료율 (%)</label>
              <input type="number" id="edit_rate" name="rate" step="0.01" min="0" max="100" value="0"
                class="w-full px-4 py-2 rounded bg-gray-700 text-white border border-gray-600 focus:outline-none focus:ring-2 focus:ring-primary">
              <p class="text-xs text-yellow-400 mt-1" id="editRateHelpText">수수료율 정보를 확인 중...</p>
            </div>
            
            <div>
              <label class="block text-sm text-gray-400 mb-2">현재 잔액</label>
              <input type="text" id="edit_current_balance" readonly
                class="w-full px-4 py-2 rounded bg-gray-700 text-gray-400 border border-gray-600">
            </div>
          </div>
        </div>
        
        <!-- 버튼 영역 -->
        <div class="flex justify-end space-x-3 pt-4 border-t border-gray-700">
          <button type="button" onclick="closeEditModal()"
            class="px-6 py-2 bg-gray-700 text-white rounded hover:bg-gray-600 transition">
            취소
          </button>
          <button type="submit"
            class="px-6 py-2 bg-primary text-white rounded hover:bg-teal-600 transition">
            수정 저장
          </button>
        </div>
      </form>
    </div>
  </div>
</main>

<script>
// 전역 변수
let treeData = [];
let allExpanded = true;
let displayIdCounter = 1;
let currentPartnerUserId = '<?= $row['user_id'] ?>';
let currentPartnerRate = <?= $row['rate'] ?>;

const kakaoAccounts = <?= json_encode($kakao_accounts) ?>;
const bankAccounts = <?= json_encode($bank_accounts) ?>;

// 페이지 로드 시 트리 데이터 가져오기
document.addEventListener('DOMContentLoaded', function() {
  fetchTreeData();
});

// 트리 데이터 가져오기 (파트너 전용)
function fetchTreeData(search = '') {
  let url = 'ajax/fetch_partner_stores.php?view=tree';
  if (search) {
    url += `&search=${encodeURIComponent(search)}`;
  }

  fetch(url)
    .then(res => res.json())
    .then(data => {
      if (data.success) {
        treeData = data.tree_data;
        displayIdCounter = 1;
        renderTreeTable();
      } else {
        console.error('데이터 로드 실패:', data.message);
        document.getElementById('treeTableBody').innerHTML = 
          '<tr><td colspan="11" class="text-center text-red-400 py-8">데이터를 불러오는 중 오류가 발생했습니다.</td></tr>';
      }
    })
    .catch(error => {
      console.error('네트워크 오류:', error);
      document.getElementById('treeTableBody').innerHTML = 
        '<tr><td colspan="11" class="text-center text-red-400 py-8">네트워크 오류가 발생했습니다.</td></tr>';
    });
}

// 검색 함수
function searchStores() {
  const search = document.getElementById('searchInput').value;
  fetchTreeData(search);
}

// 트리 테이블 렌더링 (파트너용으로 수정)
function renderTreeTable() {
  const tbody = document.getElementById('treeTableBody');
  tbody.innerHTML = '';
  
  if (treeData.length === 0) {
    tbody.innerHTML = '<tr><td colspan="11" class="text-center text-gray-400 py-8">등록된 하부 가맹점이 없습니다.</td></tr>';
    return;
  }
  
  treeData.forEach(node => {
    renderTreeRow(node, tbody, 0);
  });
}

// 트리 행 렌더링 (파트너 표시 추가)
function renderTreeRow(member, tbody, level) {
  const row = document.createElement('tr');
  row.className = 'tree-row';
  
  // 파트너 본인인 경우 특별 스타일 적용
  if (member.user_id === currentPartnerUserId) {
    row.classList.add('partner-self');
  }
  
  row.dataset.userId = member.user_id;
  row.dataset.level = level;
  
  const hasChildren = member.children && member.children.length > 0;
  
  // ID
  const idCell = document.createElement('td');
  idCell.textContent = displayIdCounter++;
  row.appendChild(idCell);
  
  // 단계 (파트너를 기준으로 상대적 표시)
  const levelCell = document.createElement('td');
  if (member.user_id === currentPartnerUserId) {
    levelCell.innerHTML = `<span class="level-badge level-1">본인</span>`;
  } else {
    const relativeLevel = level + 1; // 파트너 기준으로 1단계부터 시작
    levelCell.innerHTML = `<span class="level-badge level-${Math.min(relativeLevel, 5)}">하부${relativeLevel}단계</span>`;
  }
  row.appendChild(levelCell);
  
  // 가맹점 아이디 (트리 구조 포함)
  const userIdCell = document.createElement('td');
  const indent = level * 24;
  userIdCell.innerHTML = `
    <div style="padding-left: ${indent}px; display: flex; align-items: center;">
      <span class="tree-toggle ${!allExpanded && hasChildren ? 'collapsed' : ''}" 
            onclick="toggleNode(this, '${member.user_id}')"
            style="${hasChildren ? 'cursor: pointer;' : 'visibility: hidden;'}">
        ${hasChildren ? '▼' : ''}
      </span>
      <span class="text-white font-medium">${escapeHtml(member.user_id)}</span>
      ${member.user_id === currentPartnerUserId ? '<span class="ml-2 text-xs bg-purple-600 text-white px-2 py-0.5 rounded">본인</span>' : ''}
    </div>
  `;
  row.appendChild(userIdCell);
  
  // 가맹점명 (드롭다운 - 본인 제외)
  const userNameCell = document.createElement('td');
  if (member.user_id === currentPartnerUserId) {
    userNameCell.innerHTML = `<span class="text-white font-medium">${escapeHtml(member.user_name)}</span>`;
  } else {
    userNameCell.innerHTML = `
      <div class="dropdown-container">
        <button onclick="toggleDropdown('${escapeHtml(member.user_id)}')" 
                class="text-left hover:text-primary transition cursor-pointer flex items-center justify-between w-full group">
          <span>${escapeHtml(member.user_name)}</span>
          <svg class="w-4 h-4 ml-2 text-gray-400 group-hover:text-primary transition-transform dropdown-arrow-${member.user_id}" 
              fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
          </svg>
        </button>
        <div id="dropdown-${member.user_id}" 
            class="dropdown-menu hidden w-48 bg-gray-800 border border-gray-700 rounded-lg shadow-xl">
          <a href="#" onclick="handleAddSubMember('${escapeHtml(member.user_id)}', '${escapeHtml(member.user_name)}'); return false;" 
            class="block px-4 py-3 text-sm text-gray-300 hover:bg-gray-700 hover:text-white transition rounded-b-lg border-t border-gray-700">
            <span class="flex items-center">
              <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                      d="M18 9v3m0 0v3m0-3h3m-3 0h-3m-2-5a4 4 0 11-8 0 4 4 0 018 0zM3 20a6 6 0 0112 0v1H3v-1z"></path>
              </svg>
              하부추가
            </span>
          </a>
        </div>
      </div>
    `;
  }
  row.appendChild(userNameCell);
  
  // 가맹점 코드
  const codeCell = document.createElement('td');
  codeCell.innerHTML = `<span class="text-primary">${escapeHtml(member.code)}</span>`;
  row.appendChild(codeCell);

  // 카카오 연결계정
  const kakaoCell = document.createElement('td');
  if (member.kakao_account_id && member.kakao_account_id !== '0') {
    kakaoCell.innerHTML = `
      <div class="text-sm">
        <div class="text-yellow-400">${escapeHtml(member.kakao_account_name || '')}</div>
        <div class="text-gray-400 text-xs">${escapeHtml(member.kakao_tid || '')}</div>
      </div>
    `;
  } else {
    kakaoCell.innerHTML = '<span class="text-gray-500">-</span>';
  }
  row.appendChild(kakaoCell);

  // 은행 연결계정
  const bankCell = document.createElement('td');
  if (member.bank_account_id && member.bank_account_id !== '0') {
    bankCell.innerHTML = `
      <div class="text-sm">
        <div class="text-blue-400">${escapeHtml(member.bank_account_bank || '')}</div>
        <div class="text-gray-400 text-xs">${escapeHtml(member.bank_account_holder || '')} (${escapeHtml(member.bank_account_number || '')})</div>
      </div>
    `;
  } else {
    bankCell.innerHTML = '<span class="text-gray-500">-</span>';
  }
  row.appendChild(bankCell);
  
  // 수수료율
  const rateCell = document.createElement('td');
  rateCell.textContent = member.rate + '%';
  row.appendChild(rateCell);
  
  // 잔액
  const amountCell = document.createElement('td');
  amountCell.className = 'text-right';
  amountCell.textContent = Number(member.amount || 0).toLocaleString() + '원';
  row.appendChild(amountCell);
  
  // 등록일
  const dateCell = document.createElement('td');
  dateCell.className = 'text-gray-400';
  dateCell.textContent = member.created_at || '-';
  row.appendChild(dateCell);
  
  // 관리 버튼 (본인 제외)
  const actionCell = document.createElement('td');
  if (member.user_id !== currentPartnerUserId) {
    actionCell.innerHTML = '<span class="text-gray-500">-</span>';
  }
  row.appendChild(actionCell);
  
  tbody.appendChild(row);
  
  // 자식 노드들 렌더링
  if (hasChildren) {
    member.children.forEach(child => {
      renderTreeRow(child, tbody, level + 1);
    });
  }
}

// 노드 토글
function toggleNode(toggleElement, userId) {
  toggleElement.classList.toggle('collapsed');
  
  // 해당 노드의 모든 자식 행 찾기
  const allRows = document.querySelectorAll('#treeTableBody tr');
  let isParent = false;
  let parentLevel = -1;
  
  allRows.forEach(row => {
    if (row.dataset.userId === userId) {
      isParent = true;
      parentLevel = parseInt(row.dataset.level);
    } else if (isParent && parseInt(row.dataset.level) > parentLevel) {
      // 자식 행 토글
      if (toggleElement.classList.contains('collapsed')) {
        row.style.display = 'none';
      } else {
        row.style.display = '';
      }
    } else if (isParent && parseInt(row.dataset.level) <= parentLevel) {
      // 같은 레벨이거나 상위 레벨에 도달하면 중단
      isParent = false;
    }
  });
}

// 모두 펼치기
function expandAll() {
  allExpanded = true;
  document.querySelectorAll('.tree-toggle').forEach(el => {
    el.classList.remove('collapsed');
  });
  document.querySelectorAll('#treeTableBody tr').forEach(row => {
    row.style.display = '';
  });
}

// 모두 닫기
function collapseAll() {
  allExpanded = false;
  document.querySelectorAll('.tree-toggle').forEach(el => {
    if (el.textContent.trim()) {
      el.classList.add('collapsed');
    }
  });
  
  // 레벨 1이 아닌 모든 행 숨기기
  document.querySelectorAll('#treeTableBody tr').forEach(row => {
    if (parseInt(row.dataset.level) > 0) {
      row.style.display = 'none';
    }
  });
}

// 드롭다운 토글 (위치 계산 포함)
function toggleDropdown(userId) {
  const dropdown = document.getElementById(`dropdown-${userId}`);
  const arrow = document.querySelector(`.dropdown-arrow-${userId}`);
  const button = arrow.closest('button');
  
  // 다른 모든 드롭다운 닫기
  document.querySelectorAll('.dropdown-menu').forEach(menu => {
    if (menu.id !== `dropdown-${userId}`) {
      menu.classList.add('hidden');
      menu.classList.remove('dropdown-up');
      const otherUserId = menu.id.replace('dropdown-', '');
      const otherArrow = document.querySelector(`.dropdown-arrow-${otherUserId}`);
      if (otherArrow) {
        otherArrow.style.transform = 'rotate(0deg)';
      }
    }
  });
  
  // 현재 드롭다운 토글
  if (dropdown.classList.contains('hidden')) {
    // 화면에서 드롭다운 위치 계산
    const buttonRect = button.getBoundingClientRect();
    const dropdownHeight = 200; // 드롭다운의 대략적인 높이
    const viewportHeight = window.innerHeight;
    const spaceBelow = viewportHeight - buttonRect.bottom;
    const spaceAbove = buttonRect.top;
    
    // 아래쪽 공간이 부족하고 위쪽에 충분한 공간이 있으면 위로 열기
    if (spaceBelow < dropdownHeight && spaceAbove > dropdownHeight) {
      dropdown.classList.add('dropdown-up');
    } else {
      dropdown.classList.remove('dropdown-up');
    }
    
    dropdown.classList.remove('hidden');
    arrow.style.transform = 'rotate(180deg)';
  } else {
    dropdown.classList.add('hidden');
    dropdown.classList.remove('dropdown-up');
    arrow.style.transform = 'rotate(0deg)';
  }
  
  event.stopPropagation();
}

// 문서 클릭시 드롭다운 닫기
document.addEventListener('click', function(event) {
  if (!event.target.closest('.dropdown-container')) {
    document.querySelectorAll('.dropdown-menu').forEach(menu => {
      menu.classList.add('hidden');
      const userId = menu.id.replace('dropdown-', '');
      const arrow = document.querySelector(`.dropdown-arrow-${userId}`);
      if (arrow) {
        arrow.style.transform = 'rotate(0deg)';
      }
    });
  }
});

// 하부 가맹점 추가 폼 제출
document.getElementById('addSubStoreForm').addEventListener('submit', function(e) {
  e.preventDefault();
  
  const kakaoAccountId = document.getElementById('kakao_account_select').value;
  const bankAccountId = document.getElementById('bank_account_select').value;
  const rate = parseFloat(this.querySelector('input[name="rate"]').value);
  
  if (kakaoAccountId && bankAccountId) {
    alert('카카오 연결계정과 은행 연결계정 중 하나만 선택할 수 있습니다.');
    return;
  }
  
  if (!kakaoAccountId && !bankAccountId) {
    alert('카카오 연결계정 또는 은행 연결계정 중 하나를 선택해주세요.');
    return;
  }
  
  if (rate < currentPartnerRate) {
    alert(`하부 가맹점의 수수료율(${rate}%)은 본인의 수수료율(${currentPartnerRate}%)보다 낮을 수 없습니다.`);
    return;
  }
  
  const formData = new FormData(this);
  
  fetch('ajax/add_sub_member.php', {
    method: 'POST',
    body: formData
  })
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      alert('하부 가맹점이 추가되었습니다.');
      this.reset();
      resetFormState();
      fetchTreeData();
    } else {
      alert(data.message || '추가 중 오류가 발생했습니다.');
    }
  })
  .catch(err => {
    console.error(err);
    alert('오류가 발생했습니다.');
  });
});

// 하부 가맹점 추가 처리
function handleAddSubMember(parentUserId, parentUserName) {
  // 드롭다운 닫기
  document.querySelectorAll('.dropdown-menu').forEach(menu => {
    menu.classList.add('hidden');
  });
  
  openSubMemberModal(parentUserId, parentUserName);
}

// 하부 가맹점 추가 모달 열기
function openSubMemberModal(parentUserId, parentUserName) {
  document.getElementById('parentUserId').value = parentUserId;
  document.getElementById('parentUserName').textContent = parentUserName;
  
  document.getElementById('subMemberForm').reset();
  
  // 상위 가맹점의 수수료율 가져오기
  fetch(`ajax/get_member_rate.php?user_id=${parentUserId}`)
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            const parentRate = parseFloat(data.rate || 0);
            const rateInput = document.querySelector('#subMemberForm input[name="rate"]');
            
            // min 속성 설정하고 기본값도 상위 수수료율로 설정
            rateInput.setAttribute('min', parentRate);
            rateInput.setAttribute('max', '100');
            rateInput.value = parentRate; // 기본값을 상위 수수료율로 설정
            
            const rateHelpText = document.getElementById('rateHelpText');
            rateHelpText.innerHTML = `상위 가맹점 수수료율: <span class="text-primary font-semibold">${parentRate}%</span> (이 값보다 높거나 같아야 합니다)`;
            
            document.getElementById('parentRate').value = parentRate;
        }
    })
    .catch(err => {
      console.error('상위 가맹점 정보 조회 실패:', err);
    });
  
  document.getElementById('subMemberModal').classList.remove('hidden');
}

// 모달 닫기
function closeSubMemberModal() {
  document.getElementById('subMemberModal').classList.add('hidden');
  document.getElementById('subMemberForm').reset();
  resetSubMemberFormState();
}

// 폼 상태 리셋
function resetFormState() {
  const kakaoSelect = document.getElementById('kakao_account_select');
  const bankSelect = document.getElementById('bank_account_select');
  
  kakaoSelect.disabled = false;
  bankSelect.disabled = false;
  
  kakaoSelect.classList.remove('opacity-50', 'cursor-not-allowed');
  bankSelect.classList.remove('opacity-50', 'cursor-not-allowed');
}

function resetSubMemberFormState() {
  const kakaoSelect = document.getElementById('sub_kakao_account_select');
  const bankSelect = document.getElementById('sub_bank_account_select');
  
  kakaoSelect.disabled = false;
  bankSelect.disabled = false;
  
  kakaoSelect.classList.remove('opacity-50', 'cursor-not-allowed');
  bankSelect.classList.remove('opacity-50', 'cursor-not-allowed');
}

// 계정 선택 처리
function handleAccountSelection(type) {
  const kakaoSelect = document.getElementById('kakao_account_select');
  const bankSelect = document.getElementById('bank_account_select');

  if (type === 'kakao') {
    if (kakaoSelect.value !== '') {
      bankSelect.value = '';
      bankSelect.disabled = true;
      bankSelect.classList.add('opacity-50', 'cursor-not-allowed');
    } else {
      bankSelect.disabled = false;
      bankSelect.classList.remove('opacity-50', 'cursor-not-allowed');
    }
  } else if (type === 'bank') {
    if (bankSelect.value !== '') {
      kakaoSelect.value = '';
      kakaoSelect.disabled = true;
      kakaoSelect.classList.add('opacity-50', 'cursor-not-allowed');
    } else {
      kakaoSelect.disabled = false;
      kakaoSelect.classList.remove('opacity-50', 'cursor-not-allowed');
    }
  }
}

function handleSubAccountSelection(type) {
  const kakaoSelect = document.getElementById('sub_kakao_account_select');
  const bankSelect = document.getElementById('sub_bank_account_select');
  
  if (type === 'kakao') {
    if (kakaoSelect.value !== '') {
      bankSelect.value = '';
      bankSelect.disabled = true;
      bankSelect.classList.add('opacity-50', 'cursor-not-allowed');
    } else {
      bankSelect.disabled = false;
      bankSelect.classList.remove('opacity-50', 'cursor-not-allowed');
    }
  } else if (type === 'bank') {
    if (bankSelect.value !== '') {
      kakaoSelect.value = '';
      kakaoSelect.disabled = true;
      kakaoSelect.classList.add('opacity-50', 'cursor-not-allowed');
    } else {
      kakaoSelect.disabled = false;
      kakaoSelect.classList.remove('opacity-50', 'cursor-not-allowed');
    }
  }
}

// 하부 가맹점 추가 폼 제출
document.getElementById('subMemberForm').addEventListener('submit', function(e) {
  e.preventDefault();
  
  const kakaoAccountId = document.getElementById('sub_kakao_account_select').value;
  const bankAccountId = document.getElementById('sub_bank_account_select').value;
  
  const rateInput = document.getElementById('subRateInput');
  const parentRate = parseFloat(document.getElementById('parentRate').value || 0);
  const inputRate = parseFloat(rateInput.value || 0);
  
  if (kakaoAccountId && bankAccountId) {
    alert('카카오 연결계정과 은행 연결계정 중 하나만 선택할 수 있습니다.');
    return;
  }
  
  if (!kakaoAccountId && !bankAccountId) {
    alert('카카오 연결계정 또는 은행 연결계정 중 하나를 선택해주세요.');
    return;
  }
  
  if (inputRate < parentRate) {
    alert(`하부 가맹점의 수수료율(${inputRate}%)은 상위 가맹점의 수수료율(${parentRate}%)보다 낮을 수 없습니다.`);
    rateInput.focus();
    return;
  }
  
  const formData = new FormData(this);
  formData.append('parent_user_id', document.getElementById('parentUserId').value);
  
  fetch('ajax/add_sub_member.php', {
    method: 'POST',
    body: formData
  })
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      alert('하부 가맹점이 추가되었습니다.');
      closeSubMemberModal();
      fetchTreeData();
    } else {
      alert(data.message || '추가 중 오류가 발생했습니다.');
    }
  })
  .catch(err => {
    console.error(err);
    alert('오류가 발생했습니다.');
  });
});

// 가맹점 수정 - 모달 열기
function editStore(userId) {
  // 가맹점 정보 가져오기
  fetch(`ajax/get_member_info.php?user_id=${userId}`)
    .then(res => res.json())
    .then(data => {
      if (data.success) {
        openEditModal(data.member);
      } else {
        alert(data.message || '가맹점 정보를 불러올 수 없습니다.');
      }
    })
    .catch(err => {
      console.error('가맹점 정보 조회 실패:', err);
      alert('가맹점 정보를 불러오는 중 오류가 발생했습니다.');
    });
}

// 수정 모달 열기
function openEditModal(member) {
  // 모달에 데이터 채우기
  document.getElementById('editUserId').textContent = member.user_id;
  document.getElementById('edit_user_id').value = member.user_id;
  document.getElementById('edit_user_name').value = member.user_name || '';
  document.getElementById('edit_code').value = member.code || '';
  document.getElementById('edit_rate').value = member.rate || 0;
  document.getElementById('edit_bank').value = member.bank || '';
  document.getElementById('edit_account_number').value = member.account_number || '';
  document.getElementById('edit_account_holder').value = member.account_holder || '';
  document.getElementById('edit_current_balance').value = Number(member.amount || 0).toLocaleString() + '원';
  
  // 카카오 계정 선택
  const kakaoSelect = document.getElementById('edit_kakao_account');
  if (kakaoSelect) {
    kakaoSelect.value = member.kakao_account_id || '';
  }
  
  // 은행 연결계정 선택
  const bankSelect = document.getElementById('edit_bank_account');
  if (bankSelect) {
    bankSelect.value = member.bank_account_id || '';
  }
  
  // 계정 선택 상태에 따라 비활성화 처리
  resetEditFormState();
  
  if (member.kakao_account_id && member.kakao_account_id !== '0' && bankSelect) {
    bankSelect.disabled = true;
    bankSelect.classList.add('opacity-50', 'cursor-not-allowed');
  } else if (member.bank_account_id && member.bank_account_id !== '0' && kakaoSelect) {
    kakaoSelect.disabled = true;
    kakaoSelect.classList.add('opacity-50', 'cursor-not-allowed');
  }
  
  // 수수료율 help text 처리
  const rateHelpText = document.getElementById('editRateHelpText');
  const rateInput = document.getElementById('edit_rate');
  
  if (rateHelpText && rateInput) {
    if (member.user_id === currentPartnerUserId) {
      // 파트너 본인인 경우
      rateInput.setAttribute('min', '0');
      rateInput.setAttribute('data-parent-rate', '0');
      rateHelpText.innerHTML = '<span class="text-blue-400">파트너 본인입니다. 자유롭게 설정 가능합니다.</span>';
    } else if (member.parent_rate !== undefined && member.parent_rate !== null) {
      // 하부 가맹점인 경우
      const parentRate = parseFloat(member.parent_rate);
      
      rateInput.setAttribute('min', parentRate);
      rateInput.setAttribute('data-parent-rate', parentRate);
      
      // 현재 값이 상위 수수료율보다 낮으면 상위 수수료율로 설정
      const currentRate = parseFloat(member.rate || 0);
      if (currentRate < parentRate) {
        rateInput.value = parentRate;
      }
      
      rateHelpText.innerHTML = `상위 가맹점 수수료율: <span class="text-primary font-semibold">${parentRate}%</span> (이 값보다 높거나 같아야 합니다)`;
    } else {
      // parent가 없는 경우 (최상위 가맹점)
      rateInput.setAttribute('min', '0');
      rateHelpText.innerHTML = '<span class="text-gray-400">최상위 가맹점입니다. 자유롭게 설정 가능합니다.</span>';
    }
  }
  
  // 모달 표시
  document.getElementById('editMemberModal').classList.remove('hidden');
}

// 수정 모달 닫기
function closeEditModal() {
  const modal = document.getElementById('editMemberModal');
  if (modal) {
    modal.classList.add('hidden');
  }
  const form = document.getElementById('editMemberForm');
  if (form) {
    form.reset();
  }
}

// 수정 폼 상태 리셋
function resetEditFormState() {
  const kakaoSelect = document.getElementById('edit_kakao_account');
  const bankSelect = document.getElementById('edit_bank_account');
  
  if (kakaoSelect) {
    kakaoSelect.disabled = false;
    kakaoSelect.classList.remove('opacity-50', 'cursor-not-allowed');
  }
  
  if (bankSelect) {
    bankSelect.disabled = false;
    bankSelect.classList.remove('opacity-50', 'cursor-not-allowed');
  }
}

// 수정 모달에서 계정 선택 처리
function handleEditAccountSelection(type) {
  const kakaoSelect = document.getElementById('edit_kakao_account');
  const bankSelect = document.getElementById('edit_bank_account');
  
  if (type === 'kakao' && kakaoSelect) {
    if (kakaoSelect.value !== '') {
      if (bankSelect) {
        bankSelect.value = '';
        bankSelect.disabled = true;
        bankSelect.classList.add('opacity-50', 'cursor-not-allowed');
      }
    } else {
      if (bankSelect) {
        bankSelect.disabled = false;
        bankSelect.classList.remove('opacity-50', 'cursor-not-allowed');
      }
    }
  } else if (type === 'bank' && bankSelect) {
    if (bankSelect.value !== '') {
      if (kakaoSelect) {
        kakaoSelect.value = '';
        kakaoSelect.disabled = true;
        kakaoSelect.classList.add('opacity-50', 'cursor-not-allowed');
      }
    } else {
      if (kakaoSelect) {
        kakaoSelect.disabled = false;
        kakaoSelect.classList.remove('opacity-50', 'cursor-not-allowed');
      }
    }
  }
}

// 수정 폼 제출
document.getElementById('editMemberForm').addEventListener('submit', function(e) {
  e.preventDefault();
  
  const formData = new FormData(this);
  
  // 유효성 검사
  const user_name = formData.get('user_name').trim();
  const code = formData.get('code').trim();
  const rate = parseFloat(formData.get('rate') || 0);
  const bank = formData.get('bank');
  const account_number = formData.get('account_number').trim();
  const account_holder = formData.get('account_holder').trim();
  const kakao_account_id = formData.get('kakao_account_id') || '0';
  const bank_account_id = formData.get('bank_account_id') || '0';
  
  // 필수 필드 검증
  if (!user_name) {
    alert('가맹점명을 입력해주세요.');
    document.getElementById('edit_user_name').focus();
    return;
  }
  
  if (!code) {
    alert('가맹점 코드를 입력해주세요.');
    document.getElementById('edit_code').focus();
    return;
  }
  
  // 카카오/은행 계정 검증
  if (kakao_account_id === '0' && bank_account_id === '0') {
    alert('카카오 연결계정 또는 은행 연결계정 중 하나를 선택해주세요.');
    return;
  }
  
  if (kakao_account_id !== '0' && bank_account_id !== '0') {
    alert('카카오 연결계정과 은행 연결계정 중 하나만 선택할 수 있습니다.');
    return;
  }
  
  // 개별 출금정보 필수 검증
  if (!bank) {
    alert('출금 은행을 선택해주세요.');
    document.getElementById('edit_bank').focus();
    return;
  }
  
  if (!account_number) {
    alert('계좌번호를 입력해주세요.');
    document.getElementById('edit_account_number').focus();
    return;
  }
  
  if (!account_holder) {
    alert('예금주를 입력해주세요.');
    document.getElementById('edit_account_holder').focus();
    return;
  }
  
  // 입력값 길이 검증
  if (user_name.length > 100) {
    alert('가맹점명은 100자 이하여야 합니다.');
    document.getElementById('edit_user_name').focus();
    return;
  }
  
  if (code.length > 20) {
    alert('가맹점 코드는 20자 이하여야 합니다.');
    document.getElementById('edit_code').focus();
    return;
  }
  
  // 수수료율 범위 검증
  if (rate < 0 || rate > 100) {
    alert('수수료율은 0~100% 사이여야 합니다.');
    document.getElementById('edit_rate').focus();
    return;
  }
  
  // 상위 가맹점 수수료율 체크 (min 속성이 설정된 경우)
  const rateInput = document.getElementById('edit_rate');
  const minRate = parseFloat(rateInput.getAttribute('min') || 0);
  if (minRate > 0 && rate < minRate) {
    alert(`수수료율은 상위 가맹점의 수수료율(${minRate}%)보다 높거나 같아야 합니다.`);
    rateInput.focus();
    return;
  }
  
  // 계좌번호 형식 검증 (숫자와 하이픈만)
  if (!/^[0-9-]+$/.test(account_number)) {
    alert('계좌번호는 숫자와 하이픈(-)만 사용할 수 있습니다.');
    document.getElementById('edit_account_number').focus();
    return;
  }
  
  // 카카오 계정 ID가 비어있으면 0으로 설정
  if (!formData.get('kakao_account_id')) {
    formData.set('kakao_account_id', '0');
  }
  
  // 은행 계정 ID가 비어있으면 0으로 설정
  if (!formData.get('bank_account_id')) {
    formData.set('bank_account_id', '0');
  }
  
  // 서버로 전송
  fetch('ajax/update_member.php', {
    method: 'POST',
    body: formData
  })
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      alert(data.message || '가맹점 정보가 수정되었습니다.');
      closeEditModal();
      fetchTreeData();
    } else {
      alert(data.message || '수정 중 오류가 발생했습니다.');
    }
  })
  .catch(err => {
    console.error('수정 오류:', err);
    alert('수정 중 오류가 발생했습니다.');
  });
});

// 가맹점 삭제
function deleteStore(userId) {
  if (!confirm('정말로 이 가맹점을 삭제하시겠습니까?')) return;
  
  fetch('ajax/delete_member.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ user_id: userId })
  })
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      alert('가맹점이 삭제되었습니다.');
      fetchTreeData();
    } else {
      alert(data.message || '삭제 중 오류가 발생했습니다.');
    }
  })
  .catch(err => {
    console.error(err);
    alert('삭제 중 오류가 발생했습니다.');
  });
}

// 지급, 회수, 거래내역 처리 함수들
function handlePayment(userId, userName) {
  document.querySelectorAll('.dropdown-menu').forEach(menu => {
    menu.classList.add('hidden');
  });
  alert(`${userName}에게 지급 처리 (구현 필요)`);
}

function handleWithdraw(userId, userName) {
  document.querySelectorAll('.dropdown-menu').forEach(menu => {
    menu.classList.add('hidden');
  });
  alert(`${userName}에서 회수 처리 (구현 필요)`);
}

function handleTransactionHistory(userId, userName) {
  document.querySelectorAll('.dropdown-menu').forEach(menu => {
    menu.classList.add('hidden');
  });
  alert(`${userName} 거래내역 조회 (구현 필요)`);
}

// HTML 이스케이프
function escapeHtml(text) {
  if (!text) return '';
  const map = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    "'": '&#039;'
  };
  return text.toString().replace(/[&<>"']/g, m => map[m]);
}

// 입력 제한 및 검증
document.querySelector('#addSubStoreForm input[name="user_id"]').addEventListener('input', function(e) {
  this.value = this.value.replace(/[^a-zA-Z0-9_]/g, '');
});

document.querySelector('#subMemberForm input[name="user_id"]').addEventListener('input', function(e) {
  this.value = this.value.replace(/[^a-zA-Z0-9_]/g, '');
});

// 계좌번호 입력 제한 (숫자와 하이픈만)
document.getElementById('edit_account_number').addEventListener('input', function(e) {
  this.value = this.value.replace(/[^0-9-]/g, '');
});

// 수정된 수수료율 입력 검증
document.getElementById('edit_rate').addEventListener('input', function(e) {
  const value = parseFloat(this.value);
  const parentRate = parseFloat(this.getAttribute('data-parent-rate') || 0);
  
  // 100% 초과 제한
  if (value > 100) {
    this.value = 100;
  }
  
  if (value < 0) {
    this.value = 0;
  }
  
  // 상위 가맹점 수수료율보다 낮으면 경고 표시
  const helpText = document.getElementById('editRateHelpText');
  if (helpText && parentRate > 0) {
    if (value < parentRate) {
      helpText.classList.add('text-red-400');
      helpText.innerHTML = `<span class="text-red-400">⚠️ 상위 가맹점 수수료율(${parentRate}%)보다 낮습니다. 저장 시 오류가 발생합니다.</span>`;
    } else {
      helpText.classList.remove('text-red-400');
      helpText.innerHTML = `상위 가맹점 수수료율: <span class="text-primary font-semibold">${parentRate}%</span> (이 값보다 높거나 같아야 합니다)`;
    }
  }
});

// 가맹점명 길이 제한 표시
document.getElementById('edit_user_name').addEventListener('input', function(e) {
  const length = this.value.length;
  const maxLength = 100;
  
  if (length > maxLength * 0.9) {
    this.classList.add('border-yellow-500');
  } else {
    this.classList.remove('border-yellow-500');
  }
});

// ESC 키로 모달 닫기
document.addEventListener('keydown', function(event) {
  if (event.key === 'Escape') {
    const subModal = document.getElementById('subMemberModal');
    const editModal = document.getElementById('editMemberModal');
    
    if (subModal && !subModal.classList.contains('hidden')) {
      closeSubMemberModal();
    } else if (editModal && !editModal.classList.contains('hidden')) {
      closeEditModal();
    }
  }
});

// 모달 외부 클릭시 닫기
document.getElementById('subMemberModal').addEventListener('click', function(event) {
  if (event.target === this) {
    closeSubMemberModal();
  }
});

document.getElementById('editMemberModal').addEventListener('click', function(event) {
  if (event.target === this) {
    closeEditModal();
  }
});
</script>

</body>
</html>