<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>카카오페이 송금</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-900 text-white min-h-screen">
    <!-- 카카오페이 송금 안내 팝업 모달 -->
    <div id="kakaoPayInfoModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div class="bg-gray-800 rounded-xl p-6 max-w-md w-full mx-4 border border-yellow-400/30 max-h-[90vh] overflow-y-auto">
            <div class="text-center mb-6">
                <div class="w-20 h-20 bg-yellow-400 rounded-full flex items-center justify-center mx-auto mb-4">
                    <span class="text-3xl">💳</span>
                </div>
                <h2 class="text-2xl font-bold text-yellow-400 mb-2">올인원 페이 입금 방식 안내</h2>
                <p class="text-gray-300 text-sm">송금하기 전에 꼭 확인해주세요</p>
            </div>
            
            <div class="space-y-4 mb-6">
                <div class="bg-gray-700 rounded-lg p-4">
                    <div class="flex items-center space-x-3 mb-3">
                        <span class="text-blue-400 text-xl">0️⃣</span>
                        <h3 class="font-semibold text-blue-400">안내받으신 입금 링크 클릭</h3>
                    </div>
                </div>

                <div class="bg-gray-700 rounded-lg p-4">
                    <div class="flex items-center space-x-3 mb-3">
                        <span class="text-purple-400 text-xl">1️⃣</span>
                        <h3 class="font-semibold text-purple-400">송금 정보 입력 시</h3>
                    </div>
                    <p class="text-white text-sm">각 가맹점별 발급 받은 가맹점 코드 입력 후 송금 정보 입력</p>
                </div>

                <div class="bg-gray-700 rounded-lg p-4">
                    <div class="flex items-center space-x-3 mb-3">
                        <span class="text-green-400 text-xl">2️⃣</span>
                        <h3 class="font-semibold text-green-400">송금 정보 입력 후</h3>
                    </div>
                    <p class="text-white text-sm">화면에 안내된 <strong class="text-yellow-400">"QR 코드"</strong> 클릭 or 하단 <strong class="text-yellow-400">"카카오페이 바로 이동"</strong> 버튼 클릭 후 카카오 계정 친구 추가</p>
                </div>

                <div class="bg-gray-700 rounded-lg p-4">
                    <div class="flex items-center space-x-3 mb-3">
                        <span class="text-orange-400 text-xl">3️⃣</span>
                        <h3 class="font-semibold text-orange-400">친구 추가 후</h3>
                    </div>
                    <p class="text-white text-sm">1:1 채팅 방 내 하단 <strong class="text-yellow-400">"+"</strong> 버튼 클릭 > <strong class="text-yellow-400">"송금"</strong> 버튼 클릭 > 송금하기</p>
                    <p class="text-red-300 text-xs mt-2">(송금 외 기타 대화 절대 불가)</p>
                </div>

                <div class="bg-red-900/30 border border-red-500/50 rounded-lg p-4">
                    <div class="flex items-center space-x-3 mb-3">
                        <span class="text-red-400 text-xl">⚠️</span>
                        <h3 class="font-semibold text-red-400">※ 유의사항</h3>
                    </div>
                    <ul class="text-red-200 text-sm space-y-2">
                        <li class="flex items-start space-x-2">
                            <span class="text-red-400 mt-0.5 font-bold">1.</span>
                            <span>송금 정보 입력 시 <strong>카카오톡 이름과 입금자 신청명, 예금주 성함이 전부 동일</strong>해야 합니다.</span>
                        </li>
                        <li class="flex items-start space-x-2">
                            <span class="text-red-400 mt-0.5 font-bold">2.</span>
                            <span>카카오톡 친구 추가 후 <strong>송금하기 외 기타 대화는 절대 불가</strong>합니다.</span>
                        </li>
                        <li class="flex items-start space-x-2">
                            <span class="text-red-400 mt-0.5 font-bold">3.</span>
                            <span>개인 정보 보호를 위해 <strong>카카오페이 계정 관련 비밀번호와 인증 절차를 철저히 관리</strong>해 주세요.</span>
                        </li>
                    </ul>
                </div>
            </div>
            
            <div class="text-center">
                <button onclick="closeKakaoPayInfoModal()" 
                        class="w-full px-6 py-4 bg-yellow-400 text-gray-900 font-bold text-lg rounded-lg hover:bg-yellow-500 transition">
                    확인했습니다
                </button>
            </div>
        </div>
    </div>

    <!-- 로딩 화면 -->
    <div id="loadingScreen" class="fixed inset-0 bg-gray-900 flex items-center justify-center z-40">
        <div class="text-center">
            <div class="w-20 h-20 bg-yellow-400 rounded-full flex items-center justify-center mx-auto mb-4">
                <span class="text-3xl">💳</span>
            </div>
            <div class="w-16 h-16 border-4 border-yellow-400 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
            <p class="text-white text-lg">카카오페이 송금 신청 중...</p>
        </div>
    </div>

    <!-- 오류 화면 -->
    <div id="errorScreen" class="fixed inset-0 bg-gray-900 flex items-center justify-center z-40 hidden">
        <div class="text-center max-w-md mx-4">
            <div class="w-20 h-20 bg-red-400 rounded-full flex items-center justify-center mx-auto mb-4">
                <span class="text-3xl">❌</span>
            </div>
            <h1 class="text-2xl font-bold text-red-400 mb-4">오류 발생</h1>
            <p id="errorMessage" class="text-white mb-6">오류가 발생했습니다.</p>
            <div class="space-y-3">
                <button onclick="retryPayment()" 
                        class="w-full bg-yellow-400 text-gray-900 font-bold py-3 px-6 rounded-lg hover:bg-yellow-500 transition">
                    다시 시도
                </button>
                <button onclick="goToPreviousPage()" 
                        class="w-full bg-gray-600 text-white font-semibold py-3 px-6 rounded-lg hover:bg-gray-700 transition">
                    이전 페이지로
                </button>
            </div>
        </div>
    </div>

    <script>
        // 전역 변수
        let paymentData = null;
        let storeData = null;
        let countdownTimer = null;
        let isTimerExpired = false;
        let kakaoPayLink = null;
        let sessionId = null;
        let statusCheckTimer = null;
        // ========== 즉시 새 거래 허용 + 중복 방지 해결책 ==========
        // 추가
        let alertShown = false;  // alert이 표시되었는지 추적
        let alertDismissed = false;  // alert이 정상적으로 닫혔는지 추적
        let pageNavigating = false;
        
        //커스텀 모달 관련
        let alertModalActive = false;
        let redirectAfterAlert = false;
        let modalBlockingRefresh = false;
        
        // ========== 중복 방지를 위한 새로운 변수들 ==========
        let isInitializing = false;  // 초기화 진행 중 플래그
        let isTransactionCreating = false;  // 거래 생성 중 플래그
        let currentTransaction = null;  // 현재 활성 거래
        let modalClosed = false;  // 모달이 한번 닫혔는지 확인

        let transactionFinalized = false; //관리자에서 승인/취소를 하고 나서 유저페이지 승인완료됐는지 확인

        // 팝업 관련 함수
        function closeKakaoPayInfoModal() {
            // 이미 모달이 닫혔으면 중복 실행 방지
            if (modalClosed) {
                console.log('모달이 이미 닫혔습니다.');
                return;
            }
            
            modalClosed = true;
            document.getElementById('kakaoPayInfoModal').style.display = 'none';
            
            // 초기화가 진행 중이 아닐 때만 실행
            if (!isInitializing) {
                initializeKakaoPay();
            }
        }
        
        function showKakaoPayInfoModal() {
            modalClosed = false;
            document.getElementById('kakaoPayInfoModal').style.display = 'flex';
        }

        // 페이지 로드 시 실행
        // DOMContentLoaded 이벤트 리스너 - 간단하게 수정
        // 페이지 로드 시 실행 - 최종 버전 0910
        document.addEventListener('DOMContentLoaded', function() {
            console.log('페이지 로드 시작');
            
            // 1. 모달 상태 체크 먼저 추가
            const modalActiveTransaction = localStorage.getItem('modalActiveTransaction');
            const modalActiveTime = localStorage.getItem('modalActiveTime');
            if (modalActiveTransaction && modalActiveTime) {
                const timeDiff = Date.now() - parseInt(modalActiveTime);
                if (timeDiff < 10000) {
                    localStorage.removeItem('modalActiveTransaction');
                    localStorage.removeItem('modalActiveTime');
                    localStorage.removeItem('transactionSessionId');
                    window.location.replace('payment_select.php');
                    return;
                } else {
                    localStorage.removeItem('modalActiveTransaction');
                    localStorage.removeItem('modalActiveTime');
                }
            }

            // 2. 기존 중복 방지 체크
            const justCompleted = sessionStorage.getItem('transactionJustCompleted');
            const completedTime = sessionStorage.getItem('completedTime');
            
            if (justCompleted && completedTime) {
                const timeDiff = Date.now() - parseInt(completedTime);
                
                if (timeDiff < 5000) {
                    console.log('5초 내 중복 접근, 즉시 리다이렉트');
                    
                    // 플래그 삭제하고 즉시 이동
                    sessionStorage.removeItem('transactionJustCompleted');
                    sessionStorage.removeItem('completedTime');
                    
                    window.location.replace('payment_select.php');
                    return;
                } else {
                    // 5초 지났으면 플래그 삭제하고 정상 진행
                    sessionStorage.removeItem('transactionJustCompleted');
                    sessionStorage.removeItem('completedTime');
                }
            }
            
            // 정상적인 거래 진행
            showKakaoPayInfoModal();
            
            sessionId = localStorage.getItem('transactionSessionId');
            if (!sessionId) {
                sessionId = 'SESSION_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
                localStorage.setItem('transactionSessionId', sessionId);
            }
        });

        // 카카오페이 초기화
        function initializeKakaoPay() {
            // 완료된 거래 재확인
            const justCompleted = sessionStorage.getItem('transactionJustCompleted');
            if (justCompleted) {
                console.log('초기화 중 완료된 거래 감지, 중단');
                return;
            }
            
            if (isInitializing) {
                console.log('이미 초기화가 진행 중입니다.');
                return;
            }
            
            isInitializing = true;
            console.log('카카오페이 초기화 시작');
            
            // 기존 로직 그대로...
            const storedData = localStorage.getItem('paymentData');
            
            if (!storedData) {
                isInitializing = false;
                showError('결제 정보를 찾을 수 없습니다.');
                return;
            }

            try {
                paymentData = JSON.parse(storedData);
                
                if (paymentData.type !== 'kakao') {
                    isInitializing = false;
                    showError('카카오페이 정보가 아닙니다.');
                    return;
                }

                if (!paymentData.amount || !paymentData.depositorName || !paymentData.storeCode) {
                    isInitializing = false;
                    showError('필수 결제 정보가 누락되었습니다.');
                    return;
                }

                if (paymentData.storeData) {
                    storeData = paymentData.storeData;
                    proceedToKakaoPay();
                } else {
                    checkStoreInfo();
                }
                
            } catch (e) {
                console.error('Payment data parsing error:', e);
                isInitializing = false;
                showError('결제 정보 형식이 올바르지 않습니다.');
            }
        }


        // 가맹점 정보 확인
        function checkStoreInfo() {
            fetch('check_store_code.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ store_code: paymentData.storeCode })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    storeData = data.store;
                    proceedToKakaoPay();
                } else {
                    isInitializing = false;
                    showError('가맹점 정보를 찾을 수 없습니다: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Store check error:', error);
                isInitializing = false;
                showError('가맹점 정보 확인 중 오류가 발생했습니다.');
            });
        }

        // 카카오페이 API 호출
        function proceedToKakaoPay() {
            // console.log('카카오페이 API 호출 시작');
            
            // 먼저 기존 거래가 있는지 확인
            checkExistingTransaction();
        }


        // 기존 거래 확인 - 개선된 버전 0910 업뎃
        function checkExistingTransaction() {
            console.log('기존 거래 확인 중...');
            
            if (transactionFinalized) {
                isInitializing = false;
                return;
            }
            
            if (currentTransaction) {
                isInitializing = false;
                const apiData = {
                    success: true,
                    data: { transaction_id: currentTransaction },
                    kakao_pay_link: storeData.qr_link
                };
                showKakaoPayScreen(apiData);
                return;
            }
            
            fetch('get_remaining_time.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    session_id: sessionId,
                    check_existing: true
                })
            })
            .then(response => response.json())
            .then(data => {
                console.log('기존 거래 확인 결과:', data);
            if (data.store) {
                console.log("kakao_link_type:", data.store.kakao_link_type);
                console.log("qr_link:", data.store.qr_link);
            }
                
                // 서버에서 최근 완료된 거래 감지
                if (data.recent_completed) {
                    console.log('서버에서 최근 완료된 거래 감지');
                    alert('최근에 완료된 거래가 있습니다. 잠시 후 다시 시도해주세요.');
                    window.location.href = 'payment_select.php';
                    return;
                }
                
                if (data.existing_transaction) {
                    currentTransaction = data.existing_transaction.kakao_tid;
                    isInitializing = false;
                    
                    const apiData = {
                        success: true,
                        data: { transaction_id: data.existing_transaction.kakao_tid },
                        kakao_pay_link: storeData.qr_link
                    };
                    showKakaoPayScreen(apiData);
                } else {
                    createNewTransaction();
                }
            })
            .catch(error => {
                console.error('기존 거래 확인 오류:', error);
                isInitializing = false;
                showError('거래 확인 중 오류가 발생했습니다. 다시 시도해주세요.');
            });
        }

        // 새 거래 생성 - 개선된 버전
        function createNewTransaction() {
             // 완료된 거래가 있으면 새 거래 생성하지 않음
            if (transactionFinalized) {
                console.log('이미 완료된 거래가 있어 새 거래를 생성하지 않습니다.');
                isInitializing = false;
                return;
            }
            // 이미 거래 생성 중이면 중복 실행 방지
            if (isTransactionCreating) {
                // console.log('이미 거래 생성이 진행 중입니다.');
                return;
            }
            
            // 이미 거래가 있으면 새로 생성하지 않음
            if (currentTransaction) {
                // console.log('이미 활성 거래가 있어 새 거래를 생성하지 않습니다.');
                isInitializing = false;
                return;
            }
            
            isTransactionCreating = true;
            // console.log('새 거래 생성 시작');
            
            // 로딩 메시지 변경
            const loadingText = document.querySelector('#loadingScreen p');
            if (loadingText) {
                loadingText.textContent = '카카오페이 송금 요청 생성 중...';
            }

            const requestData = {
                store_id: paymentData.storeCode,
                amount: paymentData.amount,
                kakao_tid: storeData.kakao_tid || '',
                depositor_name: paymentData.depositorName,
                transfer_type: 'kakao',
                session_id: sessionId
            };

            // console.log('KakaoPay request data:', requestData);

            fetch('create_payment_request.php', {
                method: 'POST',
                headers: { 
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(requestData)
            })
            .then(response => response.text())
            .then(text => {
                // console.log('Response text:', text);
                try {
                    const data = JSON.parse(text);
                    if (data.success) {
                        currentTransaction = data.data.transaction_id;
                        // console.log('새 거래 생성 성공:', currentTransaction);
                        showKakaoPayScreen(data);
                    } else {
                        // 중복 거래 오류인 경우 기존 거래 사용
                        if (data.message && data.message.includes('이미 진행 중인 거래')) {
                            // console.log('기존 거래가 있음, 재확인');
                            // 기존 거래 다시 확인
                            isTransactionCreating = false;
                            isInitializing = false;
                            checkExistingTransaction();
                        } else {
                            showError('카카오페이 연결 실패: ' + data.message);
                        }
                    }
                } catch (e) {
                    console.error('JSON parsing error:', e);
                    showError('서버 응답 형식 오류: ' + text.substring(0, 100));
                } finally {
                    isTransactionCreating = false;
                    isInitializing = false;
                }
            })
            .catch(error => {
                console.error('Fetch error:', error);
                isTransactionCreating = false;
                isInitializing = false;
                showError('카카오페이 연결 중 오류가 발생했습니다: ' + error.message);
            });
        }

        // 카카오페이 QR 화면 표시
        function showKakaoPayScreen(apiData = null) {
            const transactionId = apiData ? apiData.data.transaction_id : 'LOCAL_' + Date.now();
            const qrLink = apiData ? apiData.kakao_pay_link : storeData.qr_link;
            
            // 전역 변수에 링크 저장
            kakaoPayLink = qrLink;
            
            // console.log('QR 화면 표시:', { transactionId, qrLink, apiData });
            
            document.body.innerHTML = `
                <div class="bg-gray-900 text-white min-h-screen py-8">
                    <div class="container mx-auto px-4 max-w-md">
                        <div class="text-center mb-8">
                            <div class="w-20 h-20 bg-yellow-400 rounded-full flex items-center justify-center mx-auto mb-4">
                                <span class="text-3xl">💳</span>
                            </div>
                            <h1 class="text-2xl font-bold text-yellow-400 mb-2">카카오페이 송금</h1>
                            <p class="text-gray-300">QR코드를 스캔하거나 링크를 클릭해주세요</p>
                            
                            <!-- 안내사항 다시보기 버튼 -->
                            <button onclick="showGuideModal()" 
                                    class="mt-3 px-4 py-2 bg-blue-600 text-white text-sm rounded-lg hover:bg-blue-700 transition">
                                📋 송금 안내사항 다시보기
                            </button>
                            
                            <!-- 10분 타이머 -->
                            <div class="bg-red-600/20 border border-red-500/50 rounded-lg p-3 mt-4 mb-4">
                                <div class="flex items-center justify-center space-x-2">
                                    <span class="text-red-400 text-lg">⏰</span>
                                    <span class="text-red-300 text-sm">송금 제한시간:</span>
                                    <span id="countdown" class="text-red-200 font-mono text-lg font-bold">10:00</span>
                                </div>
                                <p class="text-red-300 text-xs mt-1">시간 초과 시 자동으로 신청이 취소됩니다</p>
                            </div>
                        </div>

                        <!-- 송금 정보 -->
                        <div class="bg-gray-800 rounded-xl p-6 mb-6">
                            <h3 class="text-lg font-semibold mb-4 text-yellow-400">송금 정보</h3>
                            <div class="space-y-3">
                                <div class="flex justify-between">
                                    <span class="text-gray-300">가맹점</span>
                                    <span class="text-white font-semibold">${storeData.user_name}</span>
                                </div>
                                <div class="flex justify-between">
                                    <span class="text-gray-300">입금자</span>
                                    <span class="text-white font-semibold">${paymentData.depositorName}</span>
                                </div>
                                <div class="flex justify-between">
                                    <span class="text-gray-300">금액</span>
                                    <span class="text-white font-semibold text-xl">${paymentData.amount.toLocaleString()}원</span>
                                </div>
                                <div class="flex justify-between">
                                    <span class="text-gray-300">거래번호</span>
                                    <span class="text-white font-mono text-sm">${transactionId}</span>
                                </div>
                            </div>
                        </div>

                        <!-- QR 코드 및 링크 -->
                        <div class="bg-gradient-to-r from-yellow-500 to-yellow-600 rounded-xl p-6 mb-6">
                            <h3 class="text-lg font-semibold mb-4 text-gray-900">카카오페이 송금</h3>
                            
                            <!-- QR 코드 영역 -->
                            <div class="bg-white rounded-lg p-4 mb-4 text-center">
                                <div id="qrCodeContainer" class="w-48 h-48 bg-gray-100 rounded-lg mx-auto flex items-center justify-center mb-4 cursor-pointer hover:opacity-80 transition-opacity" onclick="openKakaoPayLink()">
                                    <div class="text-center">
                                        <div class="text-4xl mb-2">⏳</div>
                                        <p class="text-gray-600 text-sm">QR 코드 생성중...</p>
                                    </div>
                                </div>
                                <p class="text-gray-600 text-sm">카카오페이 앱으로 QR코드를 스캔하거나 클릭하세요</p>
                            </div>
                            
                            <!-- 카카오페이 링크 버튼 -->
                            <div class="space-y-3 mb-4">
                                <button onclick="openKakaoPayLink()" 
                                        class="w-full bg-yellow-500 text-gray-900 font-bold py-4 px-6 rounded-xl hover:bg-yellow-600 transition text-lg">
                                    📱 카카오페이로 바로 이동
                                </button>
                            </div>
                            
                            <!-- 이전 페이지로 이동 버튼 -->
                            <div class="space-y-3">
                                <button onclick="goToPreviousPage()" 
                                        class="w-full bg-gray-900 text-yellow-400 font-bold py-4 px-6 rounded-xl hover:bg-gray-800 transition text-lg">
                                    ⬅️ 이전 페이지로 이동
                                </button>
                            </div>
                        </div>

                        <!-- 안내 메시지 -->
                        <div class="bg-yellow-900/30 border border-yellow-500/50 rounded-lg p-4 mb-6">
                            <div class="flex items-start space-x-3">
                                <span class="text-yellow-400 text-xl">💡</span>
                                <div>
                                    <h4 class="font-semibold text-yellow-400 mb-2">송금 안내</h4>
                                    <ul class="text-sm text-yellow-200 space-y-1">
                                        <li>• QR코드를 카카오페이 앱으로 스캔해주세요</li>
                                        <li>• 입금자명을 <strong>${paymentData.depositorName}</strong>로 정확히 입력해주세요</li>
                                        <li>• 금액은 <strong>${paymentData.amount.toLocaleString()}원</strong> 정확히 송금해주세요</li>
                                        <li>• <strong>10분 이내</strong>에 송금을 완료해주세요</li>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <!-- 완료 버튼 -->
                        <div class="space-y-3">
                            <button id="confirmButton" onclick="confirmKakaoPayTransfer('${transactionId}')" 
                                    class="w-full bg-green-500 text-white font-bold py-4 px-6 rounded-xl hover:bg-green-600 transition text-lg">
                                송금 완료
                            </button>
                            <button onclick="goToPreviousPage()" 
                                    class="w-full bg-gray-600 text-white font-semibold py-3 px-6 rounded-xl hover:bg-gray-700 transition">
                                이전 페이지로
                            </button>
                        </div>
                    </div>
                </div>

                <!-- 안내사항 모달 (다시보기용) -->
                <div id="guideModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
                    <div class="bg-gray-800 rounded-xl p-6 max-w-md w-full mx-4 border border-yellow-400/30 max-h-[90vh] overflow-y-auto">
                        <div class="text-center mb-6">
                            <div class="w-20 h-20 bg-yellow-400 rounded-full flex items-center justify-center mx-auto mb-4">
                                <span class="text-3xl">💳</span>
                            </div>
                            <h2 class="text-2xl font-bold text-yellow-400 mb-2">올인원 페이 입금 방식 안내</h2>
                            <p class="text-gray-300 text-sm">송금 방법을 다시 확인해주세요</p>
                        </div>
                        
                        <div class="space-y-4 mb-6">
                            <div class="bg-gray-700 rounded-lg p-4">
                                <div class="flex items-center space-x-3 mb-3">
                                    <span class="text-blue-400 text-xl">0️⃣</span>
                                    <h3 class="font-semibold text-blue-400">안내받으신 입금 링크 클릭</h3>
                                </div>
                            </div>

                            <div class="bg-gray-700 rounded-lg p-4">
                                <div class="flex items-center space-x-3 mb-3">
                                    <span class="text-purple-400 text-xl">1️⃣</span>
                                    <h3 class="font-semibold text-purple-400">송금 정보 입력 시</h3>
                                </div>
                                <p class="text-white text-sm">각 가맹점별 발급 받은 가맹점 코드 입력 후 송금 정보 입력</p>
                            </div>

                            <div class="bg-gray-700 rounded-lg p-4">
                                <div class="flex items-center space-x-3 mb-3">
                                    <span class="text-green-400 text-xl">2️⃣</span>
                                    <h3 class="font-semibold text-green-400">송금 정보 입력 후</h3>
                                </div>
                                <p class="text-white text-sm">화면에 안내된 <strong class="text-yellow-400">"QR 코드"</strong> 클릭 or 하단 <strong class="text-yellow-400">"카카오페이 바로 이동"</strong> 버튼 클릭 후 카카오 계정 친구 추가</p>
                            </div>

                            <div class="bg-gray-700 rounded-lg p-4">
                                <div class="flex items-center space-x-3 mb-3">
                                    <span class="text-orange-400 text-xl">3️⃣</span>
                                    <h3 class="font-semibold text-orange-400">친구 추가 후</h3>
                                </div>
                                <p class="text-white text-sm">1:1 채팅 방 내 하단 <strong class="text-yellow-400">"+"</strong> 버튼 클릭 > <strong class="text-yellow-400">"송금"</strong> 버튼 클릭 > 송금하기</p>
                                <p class="text-red-300 text-xs mt-2">(송금 외 기타 대화 절대 불가)</p>
                            </div>

                            <div class="bg-red-900/30 border border-red-500/50 rounded-lg p-4">
                                <div class="flex items-center space-x-3 mb-3">
                                    <span class="text-red-400 text-xl">⚠️</span>
                                    <h3 class="font-semibold text-red-400">※ 유의사항</h3>
                                </div>
                                <ul class="text-red-200 text-sm space-y-2">
                                    <li class="flex items-start space-x-2">
                                        <span class="text-red-400 mt-0.5 font-bold">1.</span>
                                        <span>송금 정보 입력 시 <strong>카카오톡 이름과 입금자 신청명, 예금주 성함이 전부 동일</strong>해야 합니다.</span>
                                    </li>
                                    <li class="flex items-start space-x-2">
                                        <span class="text-red-400 mt-0.5 font-bold">2.</span>
                                        <span>카카오톡 친구 추가 후 <strong>송금하기 외 기타 대화는 절대 불가</strong>합니다.</span>
                                    </li>
                                    <li class="flex items-start space-x-2">
                                        <span class="text-red-400 mt-0.5 font-bold">3.</span>
                                        <span>개인 정보 보호를 위해 <strong>카카오페이 계정 관련 비밀번호와 인증 절차를 철저히 관리</strong>해 주세요.</span>
                                    </li>
                                </ul>
                            </div>
                        </div>
                        
                        <div class="text-center">
                            <button onclick="closeGuideModal()" 
                                    class="w-full px-6 py-4 bg-yellow-400 text-gray-900 font-bold text-lg rounded-lg hover:bg-yellow-500 transition">
                                확인했습니다
                            </button>
                        </div>
                    </div>
                </div>
            `;
            
            // QR 코드 생성
            generateQRCode(qrLink);
            
            // 타이머 시작
            getRemainingTimeFromDB(transactionId);

            // 상태 확인 시작
            startStatusCheck(transactionId);
        }

        // 안내사항 모달 표시/숨김
        function showGuideModal() {
            document.getElementById('guideModal').classList.remove('hidden');
        }

        function closeGuideModal() {
            document.getElementById('guideModal').classList.add('hidden');
        }

        // QR 코드 생성
        function generateQRCode(url) {
            const qrContainer = document.getElementById('qrCodeContainer');
            if (!qrContainer) {
                console.error('QR 컨테이너를 찾을 수 없습니다.');
                return;
            }
            
            const qrApiUrl = `https://api.qrserver.com/v1/create-qr-code/?size=192x192&data=${encodeURIComponent(url)}`;
            
            qrContainer.innerHTML = `
                <img src="${qrApiUrl}" 
                     alt="카카오페이 QR코드" 
                     class="w-48 h-48 rounded-lg cursor-pointer hover:opacity-80 transition-opacity"
                     onload="console.log('QR 코드 로드 완료')"
                     onerror="handleQRError()"
                     onclick="openKakaoPayLink()">
            `;
        }

        // 카카오페이 링크 열기
        function openKakaoPayLink() {
            if (kakaoPayLink) {
                // console.log('카카오페이 링크 열기:', kakaoPayLink);
                window.open(kakaoPayLink, '_blank');
            } else {
                alert('카카오페이 링크를 찾을 수 없습니다.');
            }
        }

        // QR 코드 로드 실패 시 처리
        function handleQRError() {
            const qrContainer = document.getElementById('qrCodeContainer');
            if (qrContainer) {
                qrContainer.innerHTML = `
                    <div class="text-center cursor-pointer hover:opacity-80 transition-opacity" onclick="openKakaoPayLink()">
                        <div class="text-6xl mb-2">📱</div>
                        <p class="text-gray-600 text-sm">QR 코드</p>
                        <p class="text-gray-500 text-xs">클릭하여 카카오페이로 이동</p>
                    </div>
                `;
            }
        }

        // DB에서 남은 시간 가져오기
        function getRemainingTimeFromDB(transactionId) {
            // console.log('DB에서 남은 시간 조회:', transactionId);
            
            fetch('get_remaining_time.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    transaction_id: transactionId
                })
            })
            .then(response => response.json())
            .then(data => {
                // console.log('남은 시간 응답:', data);
                
                if (data.success) {
                    if (data.is_expired) {
                        // 이미 만료된 경우
                        handleTimeExpired();
                    } else {
                        // 남은 시간으로 타이머 시작
                        startCountdown(data.remaining_seconds);
                    }
                } else {
                    console.error('남은 시간 조회 실패:', data.message);
                    // 기본 10분으로 시작
                    startCountdown(600);
                }
            })
            .catch(error => {
                console.error('남은 시간 조회 오류:', error);
                // 기본 10분으로 시작
                startCountdown(600);
            });
        }

        // 10분 카운트다운 타이머
        function startCountdown(seconds) {
            // console.log('타이머 시작:', seconds + '초');
            
            if (countdownTimer) {
                clearInterval(countdownTimer);
            }
            
            let timeLeft = seconds;
            
            countdownTimer = setInterval(() => {
                const countdownElement = document.getElementById('countdown');
                
                if (!countdownElement) {
                    console.error('countdown 엘리먼트를 찾을 수 없습니다.');
                    return;
                }
                
                const minutes = Math.floor(timeLeft / 60);
                const secs = timeLeft % 60;
                
                const display = minutes.toString().padStart(2, '0') + ':' + secs.toString().padStart(2, '0');
                countdownElement.textContent = display;
                
                if (timeLeft <= 60) {
                    countdownElement.className = 'text-red-400 font-mono text-lg font-bold animate-pulse';
                } else if (timeLeft <= 180) {
                    countdownElement.className = 'text-orange-400 font-mono text-lg font-bold';
                } else {
                    countdownElement.className = 'text-red-200 font-mono text-lg font-bold';
                }
                
                timeLeft--;
                
                if (timeLeft < 0) {
                    clearInterval(countdownTimer);
                    handleTimeExpired();
                }
            }, 1000);
        }

        // 시간 만료 처리
        function handleTimeExpired() {
            isTimerExpired = true;
            
            const confirmButton = document.getElementById('confirmButton');
            if (confirmButton) {
                confirmButton.disabled = true;
                confirmButton.className = 'w-full bg-gray-500 text-gray-300 font-bold py-4 px-6 rounded-xl cursor-not-allowed text-lg';
                confirmButton.textContent = '시간 만료';
            }
            
            const countdownElement = document.getElementById('countdown');
            if (countdownElement) {
                countdownElement.textContent = '만료';
                countdownElement.className = 'text-red-500 font-mono text-lg font-bold';
            }
            
            // 만료 시 현재 거래 초기화
            currentTransaction = null;
            
            alert('10분 제한시간이 초과되어 송금 신청이 만료되었습니다.');
        }

        // 오류 표시
        function showError(message) {
            // console.log('오류 표시:', message);
            document.getElementById('loadingScreen').classList.add('hidden');
            document.getElementById('errorScreen').classList.remove('hidden');
            document.getElementById('errorMessage').textContent = message;
            
            // 오류 시 플래그 초기화
            isInitializing = false;
            isTransactionCreating = false;
        }

        // 다시 시도 함수 - 개선된 버전
        function retryPayment() {
            // 완료된 거래가 있으면 다시 시도하지 않음
            if (transactionFinalized) {
                alert('이미 완료된 거래입니다. 새로운 거래를 원하시면 처음부터 다시 진행해주세요.');
                window.location.href = 'payment_select.php';
                return;
            }
            
            // 모든 플래그 초기화 (완료 플래그 제외)
            isInitializing = false;
            isTransactionCreating = false;
            currentTransaction = null;
            modalClosed = false;
            
            document.getElementById('errorScreen').classList.add('hidden');
            document.getElementById('loadingScreen').classList.remove('hidden');
            
            setTimeout(() => {
                initializeKakaoPay();
            }, 1000);
        }

        // 이전 페이지로 이동 - 개선된 버전
        function goToPreviousPage() {
            // 페이지 이동 시 상태 초기화
            if (statusCheckTimer) {
                clearInterval(statusCheckTimer);
            }
            if (countdownTimer) {
                clearInterval(countdownTimer);
            }
            
            // 완료된 거래가 아니면 세션 정리
            if (!transactionFinalized) {
                localStorage.removeItem('transactionSessionId');
                sessionStorage.removeItem('transactionFinalized');
                sessionStorage.removeItem('finalizedTransactionId');
            }
            
            window.location.href = 'payment_select.php';
        }

        // 카카오페이 송금 완료 확인
        function confirmKakaoPayTransfer(transactionId) {
            if (isTimerExpired) {
                alert('송금 시간이 만료되었습니다. 다시 신청해주세요.');
                return;
            }
            
            if (confirm('카카오페이 송금을 완료하셨습니까?\n완료하셨다면 "확인"을 눌러주세요.\n완료하지 않았다면 "취소"를 눌러주세요.')) {
                // 상태를 requested로 업데이트
                updateTransferStatus(transactionId);
            }
        }

        // 송금 상태 업데이트 - 개선된 버전
        let isUpdatingStatus = false;  // 상태 업데이트 중복 방지
        
        function updateTransferStatus(transactionId) {
            // 이미 업데이트 중이면 중복 실행 방지
            if (isUpdatingStatus) {
                // console.log('이미 상태 업데이트가 진행 중입니다.');
                return;
            }
            
            isUpdatingStatus = true;
            // console.log('송금 상태 업데이트 시작:', transactionId);
            
            fetch('update_transfer_status.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    transaction_id: transactionId,
                    status: 'requested'
                })
            })
            .then(response => response.json())
            .then(data => {
                // console.log('상태 업데이트 응답:', data);
                
                if (data.success) {
                    if (countdownTimer) {
                        clearInterval(countdownTimer);
                    }
                    if (statusCheckTimer) {
                        clearInterval(statusCheckTimer);
                    }
                    
                    alert('송금 완료 신청이 접수되었습니다.\n입금 확인 후 처리됩니다.');
                    goToPreviousPage();
                } else {
                    alert('이미 신청이 완료되었습니다 ');
                }
            })
            .catch(error => {
                console.error('상태 업데이트 오류:', error);
                alert('상태 업데이트 중 오류가 발생했습니다.');
            })
            .finally(() => {
                isUpdatingStatus = false;
            });
        }

        // 거래 상태 확인 함수
        // 거래 상태 확인 함수 - 개선된 버전 0910
        // checkTransactionStatus에서 완료 상태 저장 시
        function checkTransactionStatus(transactionId) {
            if (transactionFinalized) {
                return;
            }
            
            fetch('check_transaction_status.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ transaction_id: transactionId })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (data.status === 'completed') {
                        transactionFinalized = true;
                        
                        // 타이머들 정리
                        if (statusCheckTimer) clearInterval(statusCheckTimer);
                        if (countdownTimer) clearInterval(countdownTimer);
                        currentTransaction = null;
                        
                        // 중복 방지 플래그 설정 (짧은 시간)
                        sessionStorage.setItem('transactionJustCompleted', 'completed');
                        sessionStorage.setItem('completedTime', Date.now().toString());
                        
                        // alert 후 즉시 강제 이동
                        handleTransactionComplete('completed', '입금이 승인되었습니다!');
                        
                        // 세션 정리
                        localStorage.removeItem('transactionSessionId');
                        
                        // 즉시 강제 이동 (replace 사용으로 뒤로가기 방지)
                        // window.location.replace('payment_select.php');
                    }
                    else if (data.status === 'cancelled') {
                        transactionFinalized = true;
                        
                        // 타이머들 정리
                        if (statusCheckTimer) clearInterval(statusCheckTimer);
                        if (countdownTimer) clearInterval(countdownTimer);
                        currentTransaction = null;
                        
                        // 중복 방지 플래그 설정 (짧은 시간)
                        sessionStorage.setItem('transactionJustCompleted', 'cancelled');
                        sessionStorage.setItem('completedTime', Date.now().toString());
                        
                        // alert 후 즉시 강제 이동
                        handleTransactionComplete('cancelled', '입금이 취소되었습니다.');
                        
                        // 세션 정리
                        localStorage.removeItem('transactionSessionId');
                        
                        // 즉시 강제 이동 (replace 사용으로 뒤로가기 방지)
                        // window.location.replace('payment_select.php');
                    }
                }
            })
            .catch(error => {
                console.error('상태 확인 오류:', error);
            });
        }

        //커스텀 모달 관련
        function handleTransactionComplete(status, message) {
            transactionFinalized = true;
            if (statusCheckTimer) clearInterval(statusCheckTimer);
            if (countdownTimer) clearInterval(countdownTimer);
            currentTransaction = null;
            sessionStorage.setItem('transactionJustCompleted', status);
            sessionStorage.setItem('completedTime', Date.now().toString());
            localStorage.removeItem('transactionSessionId');
            const messageType = status === 'completed' ? 'success' : 'warning';
            showCustomModal(message, messageType);
            redirectAfterAlert = true;
            setTimeout(() => {
                if (alertModalActive) closeCustomModal();
                else if (redirectAfterAlert) {
                    redirectAfterAlert = false;
                    window.location.replace('payment_select.php');
                }
            }, 6000);
        }

        function startStatusCheck(transactionId) {
            // 기존 타이머가 있으면 정리
            if (statusCheckTimer) {
                clearInterval(statusCheckTimer);
            }
            
            statusCheckTimer = setInterval(() => {
                checkTransactionStatus(transactionId);
            }, 3000);
        }

        // 페이지 언로드 시 정리
        window.addEventListener('beforeunload', function(e) {
            // 거래가 완료되었으면 경고 없이 이동 허용
            if (transactionFinalized) {
                return;
            }
            
            // 진행 중인 거래가 있을 때만 경고
            if (currentTransaction && !isTimerExpired) {
                const message = '송금이 진행 중입니다. 페이지를 떠나시겠습니까?';
                e.returnValue = message;
                return message;
            }
        });

        
        window.addEventListener('focus', function() {
            const modalActiveTransaction = localStorage.getItem('modalActiveTransaction');
            const modalActiveTime = localStorage.getItem('modalActiveTime');
            if (modalActiveTransaction && modalActiveTime) {
                const timeDiff = Date.now() - parseInt(modalActiveTime);
                if (timeDiff < 10000) {
                    localStorage.removeItem('modalActiveTransaction');
                    localStorage.removeItem('modalActiveTime');
                    localStorage.removeItem('transactionSessionId');
                    window.location.replace('payment_select.php');
                }
            }
        });

        document.addEventListener('keydown', function(e) {
            if (modalBlockingRefresh || alertModalActive) {
                if (e.key === 'F5' || (e.ctrlKey && e.key === 'r') || (e.metaKey && e.key === 'r')) {
                    e.preventDefault();
                    const warningDiv = document.createElement('div');
                    warningDiv.className = 'fixed top-4 left-1/2 transform -translate-x-1/2 bg-red-600 text-white px-4 py-2 rounded-lg z-[10000]';
                    warningDiv.textContent = '처리 중입니다. 새로고침하지 마세요!';
                    document.body.appendChild(warningDiv);
                    setTimeout(() => warningDiv.remove(), 3000);
                    return false;
                }
            }
        });

        window.addEventListener('unload', function() {
            if (modalBlockingRefresh) {
                localStorage.removeItem('modalActiveTransaction');
                localStorage.removeItem('modalActiveTime');
            }
        });

        // 추가 보호: 페이지 가시성 변경 감지
        document.addEventListener('visibilitychange', function() {
            if (document.hidden && transactionFinalized) {
                // 거래가 완료된 상태에서 탭을 숨기면 강제 이동
                console.log('완료된 거래 상태에서 탭 숨김 감지');
                window.location.replace('payment_select.php');
            }
        });

        //커스텀 모달

        function showCustomModal(message, type = 'info') {
            if (alertModalActive) return;
            alertModalActive = true;
            modalBlockingRefresh = true;
            localStorage.setItem('modalActiveTransaction', 'true');
            localStorage.setItem('modalActiveTime', Date.now().toString());
            const modal = document.createElement('div');
            modal.id = 'customAlertModal';
            modal.className = 'fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-[9999]';
            const iconMap = {'success': '✅','error': '❌','warning': '⚠️','info': 'ℹ️'};
            const colorMap = {'success': 'text-green-400','error': 'text-red-400','warning': 'text-yellow-400','info': 'text-blue-400'};
            modal.innerHTML = `<div class="bg-gray-800 rounded-xl p-6 max-w-sm mx-4 text-center"><div class="w-16 h-16 bg-gray-700 rounded-full flex items-center justify-center mx-auto mb-4"><span class="text-2xl">${iconMap[type] || iconMap.info}</span></div><h3 class="text-lg font-bold ${colorMap[type] || colorMap.info} mb-3">알림</h3><p class="text-white mb-6">${message}</p><button onclick="closeCustomModal()" class="w-full bg-blue-500 text-white font-bold py-3 px-6 rounded-lg hover:bg-blue-600 transition">확인</button></div>`;
            document.body.appendChild(modal);
            setTimeout(() => {if (document.getElementById('customAlertModal')) closeCustomModal();}, 5000);
        }

        function closeCustomModal() {
            const modal = document.getElementById('customAlertModal');
            if (modal) {
                modal.remove();
                alertModalActive = false;
                modalBlockingRefresh = false;
                localStorage.removeItem('modalActiveTransaction');
                localStorage.removeItem('modalActiveTime');
                if (redirectAfterAlert) {
                    redirectAfterAlert = false;
                    window.location.replace('payment_select.php');
                }
            }
        }

    </script>
</body>
</html>