<?php
/**
 * Pushbullet WebSocket Client
 * 안드로이드 폰의 알림을 실시간으로 받아오는 PHP 클라이언트
 */

// Composer 자동 로드
require __DIR__ . '/vendor/autoload.php';
require __DIR__ . '/inc/db_connect.php';

use WebSocket\Client;

class PushbulletNotificationClient {
    private $accessToken;
    private $wsUrl;
    private $client;
    private $db;
    private $accountId;  // ← 10/22추가
    private $approvalDelay;  // ← 추가 (초 단위) 1023 추가
    
    public function __construct($accessToken, $dbConnection, $accountId = 1, $approvalDelay = 60) {
        $this->accessToken = $accessToken;
        $this->wsUrl = "wss://stream.pushbullet.com/websocket/" . $accessToken;
        $this->db = $dbConnection;
        $this->accountId = $accountId;  // ← 10/22 추가
        $this->approvalDelay = $approvalDelay;  // 1023 ← 추가
    }
    
    /**
     * WebSocket 연결 시작
     */
    public function connect() {
        echo "푸시불렛 서버에 연결 중...\n";
        echo "URL: wss://stream.pushbullet.com/websocket/***\n\n";
        
        try {
            // WebSocket 클라이언트 생성
            $this->client = new Client($this->wsUrl, [
                'timeout' => 60,
                'fragment_size' => 4096,
            ]);
            
            echo "✓ 연결 성공! 알림을 기다리는 중...\n\n";
            
            // 무한 루프로 메시지 수신
            $lastQueueCheck = time(); // 마지막 큐 체크 시간

            while (true) {
                try {
                    $message = $this->client->receive();
                    
                    if ($message) {
                        $this->handleMessage($message);
                    }
                    
                    // 10초마다 큐 처리 (1분 지난 대기 건 자동승인)
                    if (time() - $lastQueueCheck >= 10) {
                        $this->processApprovalQueue();
                        $lastQueueCheck = time();
                    }
                    
                } catch (Exception $e) {
                    echo "메시지 수신 에러: " . $e->getMessage() . "\n";
                    echo "재연결 시도 중...\n";
                    sleep(5);
                    $this->reconnect();
                }
            }
            
        } catch (Exception $e) {
            echo "연결 실패: " . $e->getMessage() . "\n";
            return false;
        }
    }
    
    /**
     * 재연결
     */
    private function reconnect() {
        try {
            if ($this->client) {
                $this->client->close();
            }
            $this->client = new Client($this->wsUrl, [
                'timeout' => 60,
                'fragment_size' => 4096,
            ]);
            echo "✓ WebSocket 재연결 성공!\n";
            
            // DB도 재연결
            $this->checkDbConnection();
            
        } catch (Exception $e) {
            echo "재연결 실패: " . $e->getMessage() . "\n";
            sleep(5);
        }
    }
    /**
        * DB 연결 확인 및 재연결
    */
    private function checkDbConnection() {
        if (!mysqli_ping($this->db)) {
            echo "[INFO] DB 연결이 끊어졌습니다. 재연결 시도 중...\n";
            
            // 기존 연결 닫기
            @mysqli_close($this->db);
            
            // 재연결 (db_connect.php를 다시 불러옴)
            global $con;
            require __DIR__ . '/inc/db_connect.php';
            $this->db = $con;
            
            if (mysqli_ping($this->db)) {
                echo "[INFO] ✓ DB 재연결 성공!\n";
                return true;
            } else {
                echo "[ERROR] ❌ DB 재연결 실패!\n";
                return false;
            }
        }
        return true;
    }
    
    /**
     * 수신된 메시지 처리
     */
    private function handleMessage($msg) {
        $data = json_decode($msg, true);
        
        if (!$data) {
            echo "JSON 파싱 실패\n";
            return;
        }
        
        $type = $data['type'] ?? '';
        
        switch($type) {
            case 'nop':
                // 연결 유지 메시지 (30초마다)
                echo "⏱ [" . date('H:i:s') . "] 연결 상태 확인 (nop)\n";
                break;
                
            case 'tickle':
                // 서버 변경 알림
                $subtype = $data['subtype'] ?? '';
                echo "🔔 [" . date('H:i:s') . "] 서버 변경 알림: {$subtype}\n";
                break;
                
            case 'push':
                // 실제 알림 데이터
                $this->handlePush($data['push'] ?? []);
                break;
                
            default:
                echo "알 수 없는 메시지 타입: {$type}\n";
                break;
        }
    }
    
    /**
     * Push 메시지 처리 (미러링된 알림)
     */
    private function handlePush($push) {
        $pushType = $push['type'] ?? '';
        
        if ($pushType === 'mirror') {
            // 미러링된 안드로이드 알림
            $this->displayNotification($push);
        } elseif ($pushType === 'dismissal') {
            // 알림 해제
            echo "🗑 알림 해제됨\n\n";
        } else {
            echo "기타 Push 타입: {$pushType}\n";
        }
    }
    
    /**
     * 알림 화면에 출력
     */
    private function displayNotification($notification) {
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
        echo "📱 새 알림 수신!\n";
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
        
        $time = date('Y-m-d H:i:s');
        $appName = $notification['application_name'] ?? '알 수 없음';
        $title = $notification['title'] ?? '';
        $body = $notification['body'] ?? '';
        $packageName = $notification['package_name'] ?? '';
        
        echo "⏰ 시간: {$time}\n";
        echo "📦 앱: {$appName}\n";
        echo "📦 패키지: {$packageName}\n";
        
        if ($title) {
            echo "📌 제목: {$title}\n";
        }
        
        if ($body) {
            echo "💬 내용: {$body}\n";
        }
        
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
        
        // 카카오페이 알림인 경우 특별 표시 + 카카오톡도 debug 추가 1008
        if (strpos($packageName, 'kakaopay') !== false || 
            strpos($appName, '카카오페이') !== false ||
            strpos($packageName, 'kakaotalk') !== false || 
            strpos($appName, '카카오톡') !== false) {

            // 어떤 앱인지 구분
            if (strpos($packageName, 'kakaotalk') !== false || strpos($appName, '카카오톡') !== false) {
                echo "💬 카카오톡 알림 감지!\n\n";
            } else {
                echo "💰 카카오페이 알림 감지!\n\n";
            }
            
            // 금액 추출
            $text = $title . ' ' . $body;
            echo "[DEBUG] 전체 텍스트: {$text}\n";
            
            if (preg_match('/(\d{1,3}(,\d{3})*|\d+)\s*원/', $text, $matches)) {
                $amount = intval(str_replace(',', '', $matches[1]));
                echo "[DEBUG] 추출된 금액: {$amount}\n";
                
                // 입금자 추출
                $depositor = null;

                // 패턴 1: 카카오톡 - 보낸사람: 
                if (preg_match('/보낸사람[:\s]*([가-힣a-zA-Z0-9*\(\)\s]+)/', $text, $matches)) {
                    $depositor = trim($matches[1]);
                    echo "[DEBUG] 패턴1(보낸사람) 매칭: {$depositor}\n";
                }
               // 패턴 2: 카카오페이 - XXX님이 (마지막 공백 뒤부터 "님이" 앞까지)
                elseif (preg_match('/\s+([가-힣a-zA-Z0-9*\(\)]+)\s*님이/', $text, $matches)) {
                    $depositor = trim($matches[1]);
                    
                    // 괄호가 있으면 괄호 안만 추출
                    if (preg_match('/\(([^)]+)\)/', $depositor, $bracketMatch)) {
                        $depositor = $bracketMatch[1];
                        echo "[DEBUG] 패턴2(님이) 매칭 - 괄호 안: {$depositor}\n";
                    } else {
                        echo "[DEBUG] 패턴2(님이) 매칭: {$depositor}\n";
                    }
                }
                // 패턴 3: 카카오톡 제목을 입금자명으로 (친구송금)
                if (!$depositor && (strpos($packageName, 'kakaotalk') !== false || strpos($appName, '카카오톡') !== false)) {
                    $depositor = trim($title);
                    echo "[DEBUG] 패턴3(카카오톡 제목) 매칭: {$depositor}\n";
                }
                
                if (!$depositor) {
                    echo "[DEBUG] 입금자 추출 실패\n";
                }
                
                if ($depositor) {
                    // 한글만 추출 (가장 안전한 방법)
                    $depositorClean = preg_replace('/[^가-힣]/', '', $depositor);

                    // 카카오페이인지 확인
                    $isKakaopay = (strpos($packageName, 'kakaopay') !== false || strpos($appName, '카카오페이') !== false);
                    $minLength = $isKakaopay ? 2 : 3;  // 카카오페이는 2글자, 카카오톡은 3글자

                    // * 패턴 매칭 (카카오페이 전용)
                    $usePattern = false;
                    $depositorPattern = $depositorClean;

                    if ($isKakaopay && strpos($depositor, '*') !== false) {
                        // * 개수 확인 및 검증
                        $starCount = substr_count($depositor, '*');
                        $cleanWithStar = preg_replace('/[^가-힣*]/', '', $depositor);
                        
                        // * 가 정확히 1개이고, 양쪽에 글자가 있는 경우만 허용
                        if ($starCount == 1 && preg_match('/^[가-힣]+\*[가-힣]+$/', $cleanWithStar)) {
                            // 양쪽 글자수 확인 (총 3글자 이상)
                            $parts = explode('*', $cleanWithStar);
                            $totalLength = mb_strlen($parts[0]) + mb_strlen($parts[1]);
                            
                            if ($totalLength >= 2) {
                                // 패턴 매칭 모드
                                $depositorPattern = str_replace('*', '%', $cleanWithStar);
                                $usePattern = true;
                                echo "[DEBUG] 패턴 모드 활성화: {$depositor} → {$depositorPattern} (양쪽 합: {$totalLength}글자)\n";
                            } else {
                                echo "[DEBUG] 패턴 거부: 양쪽 합이 3글자 미만 ({$totalLength}글자)\n";
                            }
                        } else {
                            echo "[DEBUG] 패턴 거부: * 개수({$starCount})나 형식 불일치\n";
                        }
                    }

                    if (!$usePattern) {
                        echo "[DEBUG] 변환된 입금자: {$depositorClean}\n";
                    }

                    echo "[DEBUG] 앱 종류: " . ($isKakaopay ? "카카오페이 (최소 2글자)" : "카카오톡 (최소 3글자)") . "\n";
                    
                    // 먼저 같은 금액의 대기 중인 신청 확인
                    $checkSql = "SELECT id, depositor_name, amount FROM transfer 
                                WHERE type = 'deposit' 
                                AND status IN ('requested', 'pending')
                                AND amount = ?";
                    
                    $checkStmt = mysqli_prepare($this->db, $checkSql);
                    mysqli_stmt_bind_param($checkStmt, 'd', $amount);
                    mysqli_stmt_execute($checkStmt);
                    $checkResult = mysqli_stmt_get_result($checkStmt);
                    
                    echo "[DEBUG] 같은 금액의 대기 중인 신청:\n";
                    while ($row = mysqli_fetch_assoc($checkResult)) {
                        echo "[DEBUG]   ID:{$row['id']} | {$row['depositor_name']} | {$row['amount']}\n";
                    }
                    
                  
                    // 실제 매칭 시도
                    if ($usePattern) {
                        // 패턴 매칭 (카카오페이 * 전용)
                        $sql = "SELECT *, 
                                REGEXP_REPLACE(depositor_name, '[^가-힣]', '') as name_hangul_only
                                FROM transfer 
                                WHERE type = 'deposit' 
                                AND status IN ('requested', 'pending')
                                AND amount = ? 
                                AND REGEXP_REPLACE(depositor_name, '[^가-힣]', '') LIKE ?
                                ORDER BY created_at DESC 
                                LIMIT 1";
                        
                        $stmt = mysqli_prepare($this->db, $sql);
                        mysqli_stmt_bind_param($stmt, 'ds', $amount, $depositorPattern);
                        echo "[DEBUG] 패턴 매칭 SQL 실행: amount={$amount}, pattern={$depositorPattern}\n";
                        
                    } else {
                        // 기존 매칭 로직 (완전 일치 또는 부분 포함)
                        $sql = "SELECT *, 
                                REGEXP_REPLACE(depositor_name, '[^가-힣]', '') as name_hangul_only
                                FROM transfer 
                                WHERE type = 'deposit' 
                                AND status IN ('requested', 'pending')
                                AND amount = ? 
                                AND (
                                (REGEXP_REPLACE(depositor_name, '[^가-힣]', '') = ?)
                                OR 
                                (CHAR_LENGTH(REGEXP_REPLACE(depositor_name, '[^가-힣]', '')) >= ? 
                                AND CHAR_LENGTH(?) >= ?
                                AND REGEXP_REPLACE(depositor_name, '[^가-힣]', '') LIKE CONCAT('%', ?, '%'))
                                OR 
                                (CHAR_LENGTH(REGEXP_REPLACE(depositor_name, '[^가-힣]', '')) >= ?
                                AND CHAR_LENGTH(?) >= ? 
                                AND ? LIKE CONCAT('%', REGEXP_REPLACE(depositor_name, '[^가-힣]', ''), '%'))
                                )
                                ORDER BY created_at DESC 
                                LIMIT 1";
                        
                        $stmt = mysqli_prepare($this->db, $sql);
                        mysqli_stmt_bind_param($stmt, 'dsisississ', 
                            $amount, 
                            $depositorClean, 
                            $minLength, $depositorClean, $minLength,
                            $depositorClean,
                            $minLength, $depositorClean, $minLength,
                            $depositorClean
                        );
                    }

                        mysqli_stmt_execute($stmt);
                        $result = mysqli_stmt_get_result($stmt);
                        $transfer = mysqli_fetch_assoc($result);
                    
                        // 알림 데이터를 JSON으로 저장
                        $notificationData = json_encode([
                            'account_id' => $this->accountId,  // ← 1022추가
                            'account_name' => "계정{$this->accountId}",  // ← 1022추가
                            'depositor' => $depositor,  // 원본 유지 (로그용)
                            'depositor_clean' => $depositorClean,  // 한글만 추출한 버전
                            'depositor_pattern' => $depositorPattern,  // 패턴 (있을 경우)
                            'use_pattern' => $usePattern,  // 패턴 모드 여부
                            'amount' => $amount,
                            'packageName' => $packageName ?? '',
                            'appName' => $appName ?? '',
                            'title' => $title ?? '',
                            'body' => $body ?? ''
                        ], JSON_UNESCAPED_UNICODE);

                        // 먼저 알림을 notification_queue에 저장 (매칭 여부와 상관없이)
                        $saveNotificationQuery = "INSERT INTO notification_queue 
                                                (account_id, account_name, depositor, amount, notification_data, matched, created_at) 
                                                VALUES (?, ?, ?, ?, ?, 0, NOW())";
                        $saveNotificationStmt = mysqli_prepare($this->db, $saveNotificationQuery);
                        $accountName = "계정{$this->accountId}";
                        mysqli_stmt_bind_param($saveNotificationStmt, "issds", 
                            $this->accountId, $accountName, $depositorClean, $amount, $notificationData);
                        mysqli_stmt_execute($saveNotificationStmt);
                        $notificationId = mysqli_insert_id($this->db);
                        mysqli_stmt_close($saveNotificationStmt);

                        echo "[INFO] 알림 저장됨 (Notification ID: {$notificationId})\n";

                    if ($transfer) {
                            echo "[DEBUG] 매칭 성공! ID: {$transfer['id']}\n";
                            
                            // DB에서 최신 설정 불러오기
                            $settingsQuery = "SELECT auto_approval_wait_time FROM settings LIMIT 1";
                            $settingsResult = mysqli_query($this->db, $settingsQuery);
                            $settings = mysqli_fetch_assoc($settingsResult);
                            $currentDelay = $settings['auto_approval_wait_time'] ?? $this->approvalDelay;
                            
                            $delayText = $currentDelay >= 60 
                                ? floor($currentDelay / 60) . "분" 
                                : $currentDelay . "초";
                            echo "⏰ {$delayText} 후 자동승인 예약 중...\n";
                            
                            // 큐에 추가 (설정된 시간 후 처리 예약)
                            $queuedAt = date('Y-m-d H:i:s');
                            $processAt = date('Y-m-d H:i:s', strtotime("+{$currentDelay} seconds"));
                            
                            // 이미 큐에 있는지 확인 (중복 방지)
                            $checkQueueQuery = "SELECT id FROM auto_approval_queue 
                                                WHERE transfer_id = ? AND status = 'waiting'";
                            $checkQueueStmt = mysqli_prepare($this->db, $checkQueueQuery);
                            mysqli_stmt_bind_param($checkQueueStmt, "i", $transfer['id']);
                            mysqli_stmt_execute($checkQueueStmt);
                            $checkQueueResult = mysqli_stmt_get_result($checkQueueStmt);
                            $existingQueue = mysqli_fetch_assoc($checkQueueResult);
                            mysqli_stmt_close($checkQueueStmt);
                            
                            if ($existingQueue) {
                                echo "⚠️ 이미 승인 대기 중인 건입니다. (Queue ID: {$existingQueue['id']})\n\n";
                            } else {
                                // 큐에 추가
                                $queueQuery = "INSERT INTO auto_approval_queue 
                                            (account_id, account_name, transfer_id, depositor, amount, notification_data, queued_at, process_at, status) 
                                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'waiting')";

                                $queueStmt = mysqli_prepare($this->db, $queueQuery);
                                $accountName = "계정{$this->accountId}";
                                mysqli_stmt_bind_param($queueStmt, "isisdsss", 
                                    $this->accountId, $accountName, $transfer['id'], $depositor, $amount, $notificationData, $queuedAt, $processAt
                                );
                                
                                if (mysqli_stmt_execute($queueStmt)) {
                                    $queueId = mysqli_insert_id($this->db);
                                    echo "✅ 자동승인 대기열에 추가됨\n";
                                    echo "   - Queue ID: {$queueId}\n";
                                    echo "   - Transfer ID: {$transfer['id']}\n";
                                    echo "   - 입금자: {$depositor}\n";
                                    echo "   - 금액: " . number_format($amount) . "원\n";
                                    echo "   - 처리 예정: {$processAt}\n\n";
                                    
                                    // notification_queue 매칭 완료 표시
                                    $updateNotificationQuery = "UPDATE notification_queue 
                                                            SET matched = 1, transfer_id = ?, matched_at = NOW() 
                                                            WHERE id = ?";
                                    $updateNotificationStmt = mysqli_prepare($this->db, $updateNotificationQuery);
                                    mysqli_stmt_bind_param($updateNotificationStmt, "ii", $transfer['id'], $notificationId);
                                    mysqli_stmt_execute($updateNotificationStmt);
                                    mysqli_stmt_close($updateNotificationStmt);
                                    
                                } else {
                                    echo "❌ 대기열 추가 실패: " . mysqli_error($this->db) . "\n\n";
                                }
                                mysqli_stmt_close($queueStmt);
                            }
                        } else {
                            echo "[DEBUG] 매칭 실패 - 조건을 만족하는 신청 없음\n";
                            echo "[INFO] 알림은 저장되었으며, 신청 내역이 들어오면 재매칭 시도됩니다.\n\n";
                        }
                }
            } else {
                echo "[DEBUG] 금액 추출 실패\n\n";
            }
        }
        
        // 알림 데이터를 로그 파일에 저장
        $this->saveToLog($notification);
    }

        /**
     * 대기 중인 자동승인 처리 (1분 경과한 것들)
     * 10초마다 호출되어 처리 시간이 된 큐를 확인하고 자동승인 처리
     */
    private function processApprovalQueue() {
         // DB 연결 확인
        if (!$this->checkDbConnection()) {
            echo "[ERROR] DB 연결 실패로 큐 처리 건너뜀\n";
            return;
        }
        $now = date('Y-m-d H:i:s');

        // ===== 디버그: 현재 시간 출력 =====
        echo "\n[DEBUG] processApprovalQueue 실행 - 현재 시간: {$now}\n";
        
        // 처리할 시간이 된 대기 중인 항목 조회
        $queueQuery = "SELECT q.*, t.status as transfer_status 
                    FROM auto_approval_queue q
                    JOIN transfer t ON q.transfer_id = t.id
                    WHERE q.status = 'waiting' 
                    AND q.process_at <= ?
                    ORDER BY q.process_at ASC";
        
        $queueStmt = mysqli_prepare($this->db, $queueQuery);
        mysqli_stmt_bind_param($queueStmt, "s", $now);
        mysqli_stmt_execute($queueStmt);
        $queueResult = mysqli_stmt_get_result($queueStmt);
        
        while ($queue = mysqli_fetch_assoc($queueResult)) {
            echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
            echo "⏰ 자동승인 처리 시작 (Queue ID: {$queue['id']}, Transfer ID: {$queue['transfer_id']})\n";

            // ===== 디버그: 시간 비교 =====
            echo "[DEBUG] 처리 예정 시간: {$queue['process_at']}\n";
            echo "[DEBUG] 현재 시간: {$now}\n";
            $timeDiff = strtotime($now) - strtotime($queue['process_at']);
            echo "[DEBUG] 시간 차이: {$timeDiff}초\n";
            echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
            
            // transfer 상태 재확인 (수동 승인되었을 수 있음)
            if (!in_array($queue['transfer_status'], ['pending', 'requested'])) {
                echo "⚠️ 이미 처리된 건입니다. 큐에서 제거합니다.\n";
                
                // 큐 상태 업데이트
                $updateQueueQuery = "UPDATE auto_approval_queue 
                                    SET status = 'cancelled', processed_at = NOW() 
                                    WHERE id = ?";
                $updateQueueStmt = mysqli_prepare($this->db, $updateQueueQuery);
                mysqli_stmt_bind_param($updateQueueStmt, "i", $queue['id']);
                mysqli_stmt_execute($updateQueueStmt);
                mysqli_stmt_close($updateQueueStmt);
                continue;
            }
            
            // 알림 데이터 복원
            $notificationData = json_decode($queue['notification_data'], true);
            $depositor = $notificationData['depositor'] ?? '';
            
            // ===== 자동승인 처리 시작 =====
            // transfer 상태 업데이트 (중복 방지 포함)
            $updateSql = "UPDATE transfer 
                        SET status = 'completed',
                            payment_status = 'completed',
                            completed_at = NOW(),
                            updated_at = NOW(),
                            is_auto_approved = 1
                        WHERE id = ?
                        AND status IN ('pending', 'requested')";
            
            $updateStmt = mysqli_prepare($this->db, $updateSql);
            mysqli_stmt_bind_param($updateStmt, 'i', $queue['transfer_id']);
            $updateResult = mysqli_stmt_execute($updateStmt);
            $affectedRows = mysqli_stmt_affected_rows($updateStmt);
            
            // 실제로 업데이트된 경우만 처리
            if ($updateResult && $affectedRows > 0) {
                echo "🎉 Transfer 상태 업데이트 완료! (ID: {$queue['transfer_id']})\n";
                
                try {
                    // 트랜잭션 ID 생성
                    $transId = 'TXN' . date('YmdHis') . sprintf('%06d', mt_rand(0, 999999));
                    
                    // member 정보 조회
                    $memberQuery = "SELECT m.*, t.amount as deposit_amount 
                                FROM transfer t 
                                JOIN member m ON t.user_id = m.id 
                                WHERE t.id = ?";
                    $memberStmt = mysqli_prepare($this->db, $memberQuery);
                    mysqli_stmt_bind_param($memberStmt, "i", $queue['transfer_id']);
                    mysqli_stmt_execute($memberStmt);
                    $memberResult = mysqli_stmt_get_result($memberStmt);
                    $memberData = mysqli_fetch_assoc($memberResult);
                    mysqli_stmt_close($memberStmt);
                    
                    if ($memberData) {
                        $memberId = $memberData['id'];
                        $currentBalance = $memberData['amount'];
                        $depositAmount = $memberData['deposit_amount'];
                        $memberRate = $memberData['rate'] ?? 0;
                        
                        // 수수료 계산
                        $totalFeeAmount = round($depositAmount * ($memberRate / 100), 2);
                        $actualAmount = $depositAmount - $totalFeeAmount;
                        
                        // 잔액 업데이트
                        $updateBalanceQuery = "UPDATE member SET amount = amount + ? WHERE id = ?";
                        $updateBalanceStmt = mysqli_prepare($this->db, $updateBalanceQuery);
                        mysqli_stmt_bind_param($updateBalanceStmt, "di", $actualAmount, $memberId);
                        $balanceUpdated = mysqli_stmt_execute($updateBalanceStmt);
                        mysqli_stmt_close($updateBalanceStmt);
                        
                        if ($balanceUpdated) {
                            $newBalance = $currentBalance + $actualAmount;
                            
                            // transaction_history 기록
                            $historyQuery = "INSERT INTO transaction_history 
                                        (trans_id, transfer_id, user_id, type, requested_amount, 
                                        total_fee_rate, total_fee_amount, fee_rate, fee_amount, 
                                        actual_amount, balance_before, balance_after, status, 
                                        processed_by, approval_type, created_at) 
                                        VALUES (?, ?, ?, 'deposit', ?, ?, ?, ?, ?, ?, ?, ?, 'completed', 0, 'auto', NOW())";
                            
                            $historyStmt = mysqli_prepare($this->db, $historyQuery);
                            mysqli_stmt_bind_param($historyStmt, "siidddddddd", 
                                $transId, $queue['transfer_id'], $memberId, $depositAmount, 
                                $memberRate, $totalFeeAmount, $memberRate, $totalFeeAmount,
                                $actualAmount, $currentBalance, $newBalance
                            );
                            mysqli_stmt_execute($historyStmt);
                            mysqli_stmt_close($historyStmt);
                            
                            // transfer에 trans_id 업데이트
                            $updateTransIdQuery = "UPDATE transfer SET trans_id = ? WHERE id = ?";
                            $updateTransIdStmt = mysqli_prepare($this->db, $updateTransIdQuery);
                            mysqli_stmt_bind_param($updateTransIdStmt, "si", $transId, $queue['transfer_id']);
                            mysqli_stmt_execute($updateTransIdStmt);
                            mysqli_stmt_close($updateTransIdStmt);
                            
                            echo "✅ 입금 승인 성공!\n";
                            echo "   - Transaction ID: {$transId}\n";
                            echo "   - 입금액: " . number_format($depositAmount) . "원\n";
                            echo "   - 수수료: " . number_format($totalFeeAmount) . "원\n";
                            echo "   - 실지급: " . number_format($actualAmount) . "원\n";
                            echo "   - 잔액: " . number_format($currentBalance) . " → " . number_format($newBalance) . "원\n\n";

                            // all_log 테이블에 자동승인 로그 기록
                            $logDetail = "자동입금승인(1분지연) - Trans ID: {$transId}, Transfer ID: {$queue['transfer_id']}, " .
                                        "회원: {$memberData['user_name']}({$memberData['user_id']}), " .
                                        "입금자: {$depositor}, 입금액: " . number_format($depositAmount) . "원, " .
                                        "수수료: " . number_format($totalFeeAmount) . "원, " .
                                        "실지급: " . number_format($actualAmount) . "원";

                            $logOldData = json_encode([
                                'transfer_id' => $queue['transfer_id'],
                                'status' => 'requested',
                                'balance' => $currentBalance
                            ], JSON_UNESCAPED_UNICODE);

                            $logNewData = json_encode([
                                'trans_id' => $transId,
                                'status' => 'completed', 
                                'balance' => $newBalance,
                                'actual_amount' => $actualAmount
                            ], JSON_UNESCAPED_UNICODE);

                            $logQuery = "INSERT INTO all_log (user_id, ip_address, type, old_data, new_data, status, detail, created_at) 
                                        VALUES (0, 'PUSHBULLET_AUTO', 'AUTO_DEPOSIT_APPROVED', ?, ?, 1, ?, NOW())";
                                        
                            $logStmt = mysqli_prepare($this->db, $logQuery);
                            mysqli_stmt_bind_param($logStmt, "sss", $logOldData, $logNewData, $logDetail);
                            mysqli_stmt_execute($logStmt);
                            mysqli_stmt_close($logStmt);
                            
                            // 큐 상태 업데이트 (승인 완료)
                            $updateQueueQuery = "UPDATE auto_approval_queue 
                                                SET status = 'approved', processed_at = NOW() 
                                                WHERE id = ?";
                            $updateQueueStmt = mysqli_prepare($this->db, $updateQueueQuery);
                            mysqli_stmt_bind_param($updateQueueStmt, "i", $queue['id']);
                            mysqli_stmt_execute($updateQueueStmt);
                            mysqli_stmt_close($updateQueueStmt);
                            
                        } else {
                            echo "❌ 잔액 업데이트 실패\n";
                        }
                    }
                    
                } catch (Exception $e) {
                    echo "❌ 입금 처리 중 오류: " . $e->getMessage() . "\n\n";
                }
            } else {
                if ($affectedRows === 0) {
                    echo "⚠️ 이미 처리된 건입니다. (Transfer ID: {$queue['transfer_id']}) - 중복 방지\n\n";
                } else {
                    echo "❌ 상태 업데이트 실패\n\n";
                }
                
                // 큐 상태 업데이트 (취소)
                $updateQueueQuery = "UPDATE auto_approval_queue 
                                    SET status = 'cancelled', processed_at = NOW() 
                                    WHERE id = ?";
                $updateQueueStmt = mysqli_prepare($this->db, $updateQueueQuery);
                mysqli_stmt_bind_param($updateQueueStmt, "i", $queue['id']);
                mysqli_stmt_execute($updateQueueStmt);
                mysqli_stmt_close($updateQueueStmt);
            }
            // ===== 자동승인 처리 끝 =====
        }
        
        mysqli_stmt_close($queueStmt);

        // ===== 미매칭 알림 재매칭 시작 =====
        echo "\n🔄 미매칭 알림 재매칭 확인 중...\n";
        
        // 최근 5분 내 미매칭 알림 조회
        $fiveMinutesAgo = date('Y-m-d H:i:s', strtotime('-5 minutes'));
        $unmatchedQuery = "SELECT * FROM notification_queue 
                          WHERE matched = 0 
                          AND created_at >= ?
                          ORDER BY created_at ASC";
        
        $unmatchedStmt = mysqli_prepare($this->db, $unmatchedQuery);
        mysqli_stmt_bind_param($unmatchedStmt, "s", $fiveMinutesAgo);
        mysqli_stmt_execute($unmatchedStmt);
        $unmatchedResult = mysqli_stmt_get_result($unmatchedStmt);
        
        $rematchCount = 0;
        
        while ($notification = mysqli_fetch_assoc($unmatchedResult)) {
            $notificationData = json_decode($notification['notification_data'], true);
            $depositor = $notificationData['depositor'] ?? $notification['depositor'];
            $depositorClean = $notificationData['depositor_clean'] ?? preg_replace('/[^가-힣]/', '', $notification['depositor']);
            $depositorPattern = $notificationData['depositor_pattern'] ?? $depositorClean;
            $usePattern = $notificationData['use_pattern'] ?? false;
            $amount = $notification['amount'];

            // 카카오페이인지 확인
            $packageName = $notificationData['packageName'] ?? '';
            $appName = $notificationData['appName'] ?? '';
            $isKakaopay = (strpos($packageName, 'kakaopay') !== false || strpos($appName, '카카오페이') !== false);
            $minLength = $isKakaopay ? 2 : 3;

            echo "[DEBUG] 재매칭 - 원본: {$depositor}, 정제: {$depositorClean}, 패턴모드: " . ($usePattern ? "활성({$depositorPattern})" : "비활성") . ", 최소길이: {$minLength}\n";

            // transfer와 재매칭 시도
            if ($usePattern) {
                // 패턴 매칭
                $rematchSql = "SELECT * FROM transfer 
                            WHERE type = 'deposit' 
                            AND status IN ('requested', 'pending')
                            AND amount = ? 
                            AND REGEXP_REPLACE(depositor_name, '[^가-힣]', '') LIKE ?
                            ORDER BY created_at DESC 
                            LIMIT 1";

                $rematchStmt = mysqli_prepare($this->db, $rematchSql);
                mysqli_stmt_bind_param($rematchStmt, 'ds', $amount, $depositorPattern);
                
            } else {
                // 기존 매칭 로직
                $rematchSql = "SELECT * FROM transfer 
                            WHERE type = 'deposit' 
                            AND status IN ('requested', 'pending')
                            AND amount = ? 
                            AND (
                                (REGEXP_REPLACE(depositor_name, '[^가-힣]', '') = ?)
                                OR 
                                (CHAR_LENGTH(REGEXP_REPLACE(depositor_name, '[^가-힣]', '')) >= ? 
                                AND CHAR_LENGTH(?) >= ?
                                AND REGEXP_REPLACE(depositor_name, '[^가-힣]', '') LIKE CONCAT('%', ?, '%'))
                                OR 
                                (CHAR_LENGTH(REGEXP_REPLACE(depositor_name, '[^가-힣]', '')) >= ?
                                AND CHAR_LENGTH(?) >= ? 
                                AND ? LIKE CONCAT('%', REGEXP_REPLACE(depositor_name, '[^가-힣]', ''), '%'))
                            )
                            ORDER BY created_at DESC 
                            LIMIT 1";

                $rematchStmt = mysqli_prepare($this->db, $rematchSql);
                mysqli_stmt_bind_param($rematchStmt, 'dsisississ', 
                    $amount, 
                    $depositorClean, 
                    $minLength, $depositorClean, $minLength,
                    $depositorClean,
                    $minLength, $depositorClean, $minLength,
                    $depositorClean
                );
            }

            mysqli_stmt_execute($rematchStmt);
            $rematchResult = mysqli_stmt_get_result($rematchStmt);
            $transfer = mysqli_fetch_assoc($rematchResult);
            mysqli_stmt_close($rematchStmt);
            
            if ($transfer) {
                echo "\n✨ 재매칭 성공!\n";
                echo "   - Notification ID: {$notification['id']}\n";
                echo "   - Transfer ID: {$transfer['id']}\n";
                echo "   - 입금자: {$depositor}\n";
                echo "   - 금액: " . number_format($amount) . "원\n";
                
                // 이미 큐에 있는지 확인
                $checkQueueQuery = "SELECT id FROM auto_approval_queue 
                                   WHERE transfer_id = ? AND status = 'waiting'";
                $checkQueueStmt = mysqli_prepare($this->db, $checkQueueQuery);
                mysqli_stmt_bind_param($checkQueueStmt, "i", $transfer['id']);
                mysqli_stmt_execute($checkQueueStmt);
                $checkQueueResult = mysqli_stmt_get_result($checkQueueStmt);
                $existingQueue = mysqli_fetch_assoc($checkQueueResult);
                mysqli_stmt_close($checkQueueStmt);
                
                if (!$existingQueue) {
                    // DB에서 최신 설정 불러오기
                    $settingsQuery = "SELECT auto_approval_wait_time FROM settings LIMIT 1";
                    $settingsResult = mysqli_query($this->db, $settingsQuery);
                    $settings = mysqli_fetch_assoc($settingsResult);
                    $currentDelay = $settings['auto_approval_wait_time'] ?? $this->approvalDelay;
                    
                    // 큐에 추가 (설정된 시간 후 처리)
                    $queuedAt = date('Y-m-d H:i:s');
                    $processAt = date('Y-m-d H:i:s', strtotime("+{$currentDelay} seconds"));
                    
                    // notification_data에서 계정 정보 복원
                    $notifData = json_decode($notification['notification_data'], true);
                    $accountId = $notifData['account_id'] ?? 1;
                    $accountNameFromNotif = $notifData['account_name'] ?? "계정{$accountId}";

                    $queueQuery = "INSERT INTO auto_approval_queue 
                                (account_id, account_name, transfer_id, depositor, amount, notification_data, queued_at, process_at, status) 
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'waiting')";

                    $queueStmt = mysqli_prepare($this->db, $queueQuery);
                    mysqli_stmt_bind_param($queueStmt, "isisdsss", 
                        $accountId, $accountNameFromNotif, $transfer['id'], $depositor, $amount, 
                        $notification['notification_data'], $queuedAt, $processAt
                    );
                    
                    if (mysqli_stmt_execute($queueStmt)) {
                        $queueId = mysqli_insert_id($this->db);
                        echo "   ✅ 자동승인 대기열에 추가됨 (Queue ID: {$queueId}, 처리 예정: {$processAt})\n";
                        
                        // notification_queue 매칭 완료 표시
                        $updateNotificationQuery = "UPDATE notification_queue 
                                                   SET matched = 1, transfer_id = ?, matched_at = NOW() 
                                                   WHERE id = ?";
                        $updateNotificationStmt = mysqli_prepare($this->db, $updateNotificationQuery);
                        mysqli_stmt_bind_param($updateNotificationStmt, "ii", $transfer['id'], $notification['id']);
                        mysqli_stmt_execute($updateNotificationStmt);
                        mysqli_stmt_close($updateNotificationStmt);
                        
                        $rematchCount++;
                    }
                    mysqli_stmt_close($queueStmt);
                } else {
                    echo "   ⚠️ 이미 대기 중 (Queue ID: {$existingQueue['id']})\n";
                }
            }
        }
        
        mysqli_stmt_close($unmatchedStmt);
        
        if ($rematchCount > 0) {
            echo "\n✅ 재매칭 완료: {$rematchCount}건\n";
        } else {
            echo "재매칭할 알림 없음\n";
        }
        // ===== 미매칭 알림 재매칭 끝 =====
        
    }

    
    /**
     * 알림을 로그 파일에 저장
     */
    private function saveToLog($notification) {
        $logFile = __DIR__ . '/notifications3.log';
        $logData = [
            'timestamp' => date('Y-m-d H:i:s'),
            'app' => $notification['application_name'] ?? '',
            'title' => $notification['title'] ?? '',
            'body' => $notification['body'] ?? '',
            'package' => $notification['package_name'] ?? ''
        ];
        
        file_put_contents(
            $logFile, 
            json_encode($logData, JSON_UNESCAPED_UNICODE) . "\n", 
            FILE_APPEND
        );
    }
}

// 사용 예제
if (php_sapi_name() === 'cli') {
    // CLI에서 실행할 때만 동작
    
    // ⚠️ 여기에 푸시불렛 Access Token을 입력하세요
    $accessToken = 'o.gOnLkrTmxPDQsVWqF4ufgpk70USCe7n5';
    
    if ($accessToken === 'YOUR_PUSHBULLET_ACCESS_TOKEN_HERE') {
        echo "❌ 에러: Access Token을 설정해주세요!\n";
        echo "1. https://www.pushbullet.com/#settings/account 접속\n";
        echo "2. 'Access Tokens' 섹션에서 토큰 생성\n";
        echo "3. 코드의 \$accessToken 변수에 입력\n";
        exit(1);
    }
    
    // DB에서 자동승인 대기시간 불러오기
    $settingsQuery = "SELECT auto_approval_wait_time FROM settings LIMIT 1";
    $settingsResult = mysqli_query($con, $settingsQuery);
    $settings = mysqli_fetch_assoc($settingsResult);
    $approvalDelay = $settings['auto_approval_wait_time'] ?? 60;  // 기본값 60초

    echo "═══════════════════════════════════════════════\n";
    echo "  푸시불렛 알림 모니터링 시작 (계정3)\n";
    echo "  자동승인 대기시간: {$approvalDelay}초\n";
    echo "═══════════════════════════════════════════════\n\n";

    $client = new PushbulletNotificationClient($accessToken, $con, 3, $approvalDelay);  // ← 3번 계정
    
    try {
        $client->connect();
    } catch (Exception $e) {
        echo "에러: " . $e->getMessage() . "\n";
    }
} else {
    echo "이 스크립트는 CLI(명령줄)에서만 실행할 수 있습니다.\n";
}