<?php
date_default_timezone_set('Asia/Seoul');
require_once 'vendor/autoload.php';
require_once 'sms_parser.php';

use React\EventLoop\Loop;

class PushbulletDaemon {
    private $apiKey;
    private $pdo;
    private $parser;
    private $loop;
    private $lastCheckTime = 0;
    private $processedPushIds = []; // 성능용 메모리 캐시
    private $maxRetries = 3;
    private $baseRetryDelay = 5; // 초
    
    public function __construct($apiKey, $dbConfig) {
        $this->apiKey = $apiKey;
        $this->setupDatabase($dbConfig);
        $this->parser = new SMSParser();
        $this->loop = Loop::get();
        
        // 초기 실행 시에는 24시간 전부터 가져오기 (기존 SMS도 포함)
        // $this->lastCheckTime = time() - (24 * 3600);
        $this->lastCheckTime = time() - (10 * 24 * 3600);  // 10일 전부터 체크
        
        // 이미 처리된 푸시들을 DB에서 로드
        $this->loadProcessedPushIds();
    }
    
    // 은행 SMS인지 확인하는 함수 (기존 유지)
    private function isBankSMS($content) {
        $bankKeywords = [
            '입금', '출금', '잔액', '이체',
            '[카카오뱅크]', '[신한은행]', '[KB]', '[국민은행]', 
            '[우리은행]', '[하나은행]', '[농협]', '[NH농협]',
            '[기업은행]', '[IBK기업은행]', '[새마을금고]', '[토스뱅크]',
            '하나,'  // 하나은행 특별 형식
        ];
        
        foreach ($bankKeywords as $keyword) {
            if (strpos($content, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }

    private function setupDatabase($config) {
        $dsn = "mysql:host={$config['host']};dbname={$config['dbname']};charset=utf8mb4";
        $this->pdo = new PDO($dsn, $config['username'], $config['password']);
        $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // push_id 컬럼이 없다면 추가 (기존 테이블 호환성)
        $this->ensurePushIdColumn();
        
        echo "DB 연결 완료\n";
    }
    
    private function ensurePushIdColumn() {
        try {
            // push_id 컬럼 존재 여부 확인
            $stmt = $this->pdo->query("SHOW COLUMNS FROM sms_bank LIKE 'push_id'");
            if ($stmt->rowCount() == 0) {
                // push_id 컬럼이 없으면 추가
                $this->pdo->exec("ALTER TABLE sms_bank ADD COLUMN push_id VARCHAR(50) NULL AFTER content");
                echo "push_id 컬럼 추가 완료\n";
            }
        } catch (Exception $e) {
            echo "테이블 구조 확인/수정 중 오류: " . $e->getMessage() . "\n";
        }
    }
    
    private function loadProcessedPushIds() {
        try {
            // 최근 48시간 내의 처리된 push_id들을 메모리에 로드 (더 안전하게)
            $stmt = $this->pdo->prepare("SELECT push_id FROM sms_bank WHERE push_id IS NOT NULL AND regist_date >= DATE_SUB(NOW(), INTERVAL 48 HOUR)");
            $stmt->execute();
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                if (!empty($row['push_id'])) {
                    $this->processedPushIds[$row['push_id']] = true;
                }
            }
            
            echo "기존 처리된 푸시 ID " . count($this->processedPushIds) . "개 로드 완료\n";
        } catch (Exception $e) {
            echo "처리된 푸시 ID 로드 중 오류: " . $e->getMessage() . "\n";
        }
    }
    
    public function start() {
        echo "푸시불렛 데몬 시작 (개선된 안정성)...\n";
        
        // 초기 SMS 데이터 가져오기
        $this->fetchRecentSMS();
        
        // 15초마다 새 SMS 확인 (너무 자주 호출하지 않도록 조정)
        $this->loop->addPeriodicTimer(15, function() {
            echo "\n" . date('Y-m-d H:i:s') . " - SMS 확인 중...\n";
            $this->fetchRecentSMS();
        });
        
        // 2시간마다 오래된 푸시 ID 정리
        $this->loop->addPeriodicTimer(7200, function() {
            $this->cleanupOldPushIds();
        });
        
        // 6시간마다 상태 점검 및 자가 진단
        $this->loop->addPeriodicTimer(21600, function() {
            $this->performHealthCheck();
        });
        
        echo "15초마다 SMS 확인 모드로 실행 중...\n";
        $this->loop->run();
    }
    
    private function fetchRecentSMS() {
        $currentTime = time();
        
        echo "=== 디버깅 정보 ===\n";
        echo "현재 타임스탬프: $currentTime\n";
        echo "현재 시간: " . date('Y-m-d H:i:s', $currentTime) . "\n";
        echo "마지막 체크 타임스탬프: {$this->lastCheckTime}\n";
        echo "마지막 체크 시간: " . date('Y-m-d H:i:s', $this->lastCheckTime) . "\n";
        echo "==================\n";
        
        echo "체크 시간 범위: " . date('Y-m-d H:i:s', $this->lastCheckTime) . " ~ " . date('Y-m-d H:i:s', $currentTime) . "\n";
        
        // SMS 전용 API로 변경 - 먼저 디바이스 목록 가져오기
        $deviceUrl = "https://api.pushbullet.com/v2/devices";
        echo "디바이스 목록 조회: $deviceUrl\n";
        
        $deviceResponse = $this->makeRequestWithRetry($deviceUrl);
        
        if (!$deviceResponse) {
            echo "디바이스 목록 조회 실패\n";
            return;
        }
        
        $deviceData = json_decode($deviceResponse, true);
        $smsDevice = null;
        
        // SMS 가능한 디바이스 찾기
        if (isset($deviceData['devices'])) {
            foreach ($deviceData['devices'] as $device) {
                if (isset($device['has_sms']) && $device['has_sms'] === true && 
                    isset($device['active']) && $device['active'] === true) {
                    $smsDevice = $device;
                    echo "SMS 가능 디바이스 발견: " . $device['nickname'] . " (" . $device['iden'] . ")\n";
                    break;
                }
            }
        }
        
        if (!$smsDevice) {
            echo "SMS 가능한 활성 디바이스를 찾을 수 없습니다.\n";
            $this->lastCheckTime = $currentTime;
            return;
        }
        
        // SMS 스레드 목록 가져오기 (실제 SMS 데이터)
        $threadsUrl = "https://api.pushbullet.com/v2/permanents/" . $smsDevice['iden'] . "_threads";
        echo "SMS 스레드 조회: $threadsUrl\n";
        
        $threadsResponse = $this->makeRequestWithRetry($threadsUrl);
        
        if ($threadsResponse) {
            $threadsData = json_decode($threadsResponse, true);
            
            if (isset($threadsData['threads']) && count($threadsData['threads']) > 0) {
                echo "=== SMS 스레드 발견: " . count($threadsData['threads']) . "개 ===\n";
                
                $savedCount = 0;
                $skippedCount = 0;
                
                foreach ($threadsData['threads'] as $index => $thread) {
                    if (isset($thread['latest'])) {
                        $latest = $thread['latest'];
                        
                        echo "[$index] 스레드 ID: " . $thread['id'] . "\n";
                        echo "발신자: " . ($thread['recipients'][0]['name'] ?? '알수없음') . " (" . ($thread['recipients'][0]['address'] ?? '') . ")\n";
                        echo "시간: " . date('Y-m-d H:i:s', $latest['timestamp']) . " (타임스탬프: " . $latest['timestamp'] . ")\n";
                        echo "마지막 체크 시간: " . date('Y-m-d H:i:s', $this->lastCheckTime) . " (타임스탬프: " . $this->lastCheckTime . ")\n";
                        echo "내용: " . substr($latest['body'], 0, 100) . "\n";
                        
                        // 시간 체크 결과 출력
                        if ($latest['timestamp'] <= $this->lastCheckTime) {
                            echo "  → 시간 체크: 이전 메시지이므로 건너뜀\n";
                            echo "---\n";
                            continue;
                        }
                        
                        echo "  → 시간 체크: 새 메시지\n";
                        
                        // 은행 SMS인지 확인
                        if ($this->isBankSMS($latest['body'])) {
                            echo "  → 은행 SMS 감지됨\n";
                            
                            // SMS ID로 중복 체크
                            $smsId = $smsDevice['iden'] . '_' . $thread['id'] . '_' . $latest['id'];
                            
                            if ($this->isDuplicateSMS($smsId, $latest['body'], $latest['timestamp'])) {
                                echo "  → 중복 SMS 건너뜀\n";
                                $skippedCount++;
                            } else {
                                // SMS 데이터 구성
                                $smsData = [
                                    'iden' => $smsId,
                                    'created' => $latest['timestamp'],
                                    'body' => $latest['body']
                                ];
                                
                                $saved = $this->processSMS($latest['body'], $smsData);
                                if ($saved) {
                                    $savedCount++;
                                    $this->processedPushIds[$smsId] = true;
                                }
                            }
                        } else {
                            echo "  → 은행 SMS 아님, 저장하지 않음\n";
                        }
                        echo "---\n";
                    }
                }
                
                echo "새로 DB에 저장된 SMS: {$savedCount}개, 건너뛴 SMS: {$skippedCount}개\n";
            } else {
                echo "SMS 스레드 없음\n";
            }
        } else {
            echo "SMS 스레드 조회 실패\n";
        }
        
        // 정확한 시간 업데이트 (현재 시간으로 설정하여 다음에는 새로운 것만 가져오기)
        $this->lastCheckTime = $currentTime;
        echo "다음 체크 시작 시간 업데이트: " . date('Y-m-d H:i:s', $this->lastCheckTime) . "\n";
    }
    
    /**
     * SMS 전용 중복 체크 로직
     */
    private function isDuplicateSMS($smsId, $content, $timestamp) {
        // 1. 메모리 캐시에서 SMS ID 체크 (빠른 체크)
        if ($smsId && isset($this->processedPushIds[$smsId])) {
            return true;
        }
        
        // 2. DB에서 SMS ID 체크 (확실한 체크)
        if ($smsId) {
            try {
                $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM sms_bank WHERE push_id = ?");
                $stmt->execute([$smsId]);
                if ($stmt->fetchColumn() > 0) {
                    return true;
                }
            } catch (Exception $e) {
                echo "  → SMS ID 중복 체크 중 오류: " . $e->getMessage() . "\n";
            }
        }
        
        // 3. 내용과 시간으로 중복 체크 (SMS ID가 없는 경우나 추가 안전장치)
        try {
            $timeString = date('Y-m-d H:i:s', $timestamp);
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) FROM sms_bank 
                WHERE content = ? AND ABS(TIMESTAMPDIFF(SECOND, regist_date, ?)) <= 60
            ");
            $stmt->execute([$content, $timeString]);
            
            if ($stmt->fetchColumn() > 0) {
                echo "  → 내용+시간 기준 중복 감지\n";
                return true;
            }
        } catch (Exception $e) {
            echo "  → 내용+시간 중복 체크 중 오류: " . $e->getMessage() . "\n";
        }
        
        return false;
    }
    
    private function processSMS($content, $pushData = null) {
        try {
            $pushId = isset($pushData['iden']) ? $pushData['iden'] : null;
            
            $smsTime = isset($pushData['created']) ? 
                    date('Y-m-d H:i:s', $pushData['created']) : 
                    date('Y-m-d H:i:s');
            
            $this->pdo->beginTransaction();
            
            try {
                // 1️⃣ SMS 저장 (sms_test → sms_bank 변경)
                $stmt = $this->pdo->prepare("INSERT INTO sms_bank (content, regist_date, push_id) VALUES (?, ?, ?)");
                $result = $stmt->execute([$content, $smsTime, $pushId]);
                
                if (!$result) {
                    throw new Exception("SMS 저장 실패");
                }
                
                $smsId = $this->pdo->lastInsertId();
                
                // 2️⃣ ✨ 입금 SMS면 즉시 자동승인 처리 (새로 추가)
                $parsedSMS = $this->parser->parseBankSMS($content);
                
                if ($parsedSMS['transactionType'] === '입금') {
                    echo "  → 💰 입금 SMS 감지: {$parsedSMS['bank']}, 금액: " . number_format($parsedSMS['amount']) . "원, 입금자: {$parsedSMS['depositorName']}\n";  // ✨ 입금자명 추가
                    $this->autoApproveDeposit($parsedSMS, $smsId, $smsTime);
                }
                
                $this->pdo->commit();
                
                echo "  → SMS 저장 성공! 시간: $smsTime" . ($pushId ? ", ID: $pushId" : "") . "\n";
                return true;
                
            } catch (Exception $e) {
                $this->pdo->rollback();
                
                // push_id 컬럼이 없는 경우 기존 방식으로 저장
                if (strpos($e->getMessage(), 'push_id') !== false) {
                    echo "  → push_id 포함 저장 실패, 기존 방식으로 저장\n";
                    $stmt = $this->pdo->prepare("INSERT INTO sms_bank (content, regist_date) VALUES (?, ?)");
                    $result = $stmt->execute([$content, $smsTime]);
                    
                    if ($result) {
                        echo "  → SMS 저장 성공! (기존 방식) 시간: $smsTime\n";
                        return true;
                    }
                }
                
                throw $e;
            }
            
        } catch (Exception $e) {
            echo "  → SMS 처리 오류: " . $e->getMessage() . "\n";
            return false;
        }
    }
    
    private function cleanupOldPushIds() {
        echo "오래된 푸시 ID 메모리 정리 중...\n";
        
        // 48시간 이전의 푸시 ID들을 메모리에서 제거
        try {
            $stmt = $this->pdo->prepare("SELECT push_id FROM sms_bank WHERE push_id IS NOT NULL AND regist_date >= DATE_SUB(NOW(), INTERVAL 48 HOUR)");
            $stmt->execute();
            
            $currentValidIds = [];
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                if (!empty($row['push_id'])) {
                    $currentValidIds[$row['push_id']] = true;
                }
            }
            
            $oldCount = count($this->processedPushIds);
            $this->processedPushIds = $currentValidIds;
            $newCount = count($this->processedPushIds);
            
            echo "푸시 ID 정리 완료: {$oldCount}개 → {$newCount}개\n";
            
        } catch (Exception $e) {
            echo "푸시 ID 정리 중 오류: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * 재시도 로직이 포함된 API 요청
     */
    private function makeRequestWithRetry($url) {
        $lastError = '';
        
        for ($retry = 0; $retry < $this->maxRetries; $retry++) {
            $response = $this->makeRequest($url);
            
            if ($response !== false) {
                // 성공
                if ($retry > 0) {
                    echo "API 호출 성공 (재시도 {$retry}회 후)\n";
                }
                return $response;
            }
            
            // 실패 시 재시도
            if ($retry < $this->maxRetries - 1) {
                $delay = $this->baseRetryDelay * pow(2, $retry); // 지수적 백오프
                echo "API 호출 실패, {$delay}초 후 재시도 (" . ($retry + 1) . "/{$this->maxRetries})\n";
                sleep($delay);
            }
        }
        
        echo "모든 재시도 실패\n";
        return false;
    }
    
    private function makeRequest($url) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Access-Token: ' . $this->apiKey,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // 연결 타임아웃 추가
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_USERAGENT, 'PushbulletDaemon/1.0'); // User-Agent 추가
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            echo "CURL 오류: $error\n";
            return false;
        }
        
        if ($httpCode === 200) {
            return $response;
        }
        
        echo "API 요청 실패: HTTP $httpCode\n";
        if ($httpCode === 429) {
            echo "API 요청 한도 초과 - 더 긴 지연 후 재시도\n";
            sleep(60); // Rate limit 시 1분 대기
        } elseif ($httpCode >= 500) {
            echo "서버 오류 - 재시도 가능\n";
        } elseif ($httpCode === 401) {
            echo "인증 오류 - API 키 확인 필요\n";
        }
        
        return false;
    }
    
    /**
     * 상태 점검 및 자가 진단
     */
    private function performHealthCheck() {
        echo "\n=== 데몬 상태 점검 시작 ===\n";
        
        try {
            // 1. DB 연결 상태 확인
            $stmt = $this->pdo->query("SELECT 1");
            echo "✓ DB 연결 상태: 정상\n";
            
            // 2. 최근 SMS 수신 확인
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) as recent_count, MAX(regist_date) as last_time
                FROM sms_bank 
                WHERE regist_date >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
            ");
            $stmt->execute();
            $result = $stmt->fetch();
            
            echo "✓ 최근 1시간 SMS 수: " . $result['recent_count'] . "개\n";
            echo "✓ 마지막 SMS 시간: " . ($result['last_time'] ?: '없음') . "\n";
            
            // 3. 메모리 사용량 확인
            $memoryUsage = memory_get_usage(true);
            $memoryMB = round($memoryUsage / 1024 / 1024, 2);
            echo "✓ 메모리 사용량: {$memoryMB} MB\n";
            
            // 4. 처리된 푸시 ID 수 확인
            echo "✓ 메모리 캐시된 푸시 ID 수: " . count($this->processedPushIds) . "개\n";
            
            // 5. API 연결 테스트
            $testResponse = $this->makeRequest("https://api.pushbullet.com/v2/users/me");
            if ($testResponse) {
                echo "✓ Pushbullet API 연결: 정상\n";
            } else {
                echo "✗ Pushbullet API 연결: 실패\n";
            }
            
        } catch (Exception $e) {
            echo "✗ 상태 점검 중 오류: " . $e->getMessage() . "\n";
        }
        
        echo "=== 데몬 상태 점검 완료 ===\n\n";
    }

    /**
    * 입금 SMS 즉시 자동승인 처리
    */
    private function autoApproveDeposit($parsedSMS, $smsId, $smsTime) {
        try {
            $amount = $parsedSMS['amount'];
            $depositorName = $parsedSMS['depositorName'] ?? null;  // ✨ 추가
            
            if (!$depositorName || $depositorName === '알수없음') {  // ✨ 추가
                echo "  → ⚠️  입금자명을 알 수 없어 자동승인 불가\n";
                return;
            }
            
            // ✨ 입금자명도 체크!
            $stmt = $this->pdo->prepare("
                SELECT * FROM transfer 
                WHERE type = 'deposit'
                AND status = 'requested'
                AND amount = ?
                AND depositor_name = ?  // ✨ 추가
                AND created_at >= DATE_SUB(?, INTERVAL 10 MINUTE)
                ORDER BY created_at DESC
                LIMIT 1
            ");
            
            $stmt->execute([$amount, $depositorName, $smsTime]);  // ✨ 파라미터 추가
            $deposit = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$deposit) {
                echo "  → ⚠️  매칭되는 입금 신청 없음 (금액: " . number_format($amount) . "원)\n";
                return;
            }
            
            // ✅ 매칭 성공! 즉시 승인 처리
            echo "  → ✅ 매칭 성공! Transfer ID: {$deposit['id']}, 입금자: {$deposit['depositor_name']}\n";
            
            // 1. transfer 상태 변경
            $updateStmt = $this->pdo->prepare("
                UPDATE transfer 
                SET status = 'completed',
                    approval_type = 'auto',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $updateStmt->execute([$deposit['id']]);
            
            // 2. 회원 잔액 업데이트
            $balanceStmt = $this->pdo->prepare("
                UPDATE member 
                SET balance = balance + ?
                WHERE id = ?
            ");
            $balanceStmt->execute([$amount, $deposit['user_id']]);
            
            echo "  → 💳 자동승인 완료! 금액: " . number_format($amount) . "원 → 잔액 반영\n";
            
        } catch (Exception $e) {
            echo "  → ❌ 자동승인 오류: " . $e->getMessage() . "\n";
            // 자동승인 실패해도 SMS는 저장되어야 하므로 throw 하지 않음
        }
    }
}
?>