<?php

class SMSParser {
    
    public function parseBankSMS($smsBody) {
        $lines = array_map('trim', explode("\n", trim($smsBody)));
        
        // 은행 식별
        $bank = $this->identifyBank($smsBody);
        
        $accountHolder = '알수없음';
        $accountNumber = '알수없음';
        $transactionType = null;
        $amount = 0;
        $balance = 0;
        $memo = '';
        
        // 은행별 파싱 로직
        switch ($bank) {
            case '카카오뱅크':
                return $this->parseKakaoBank($smsBody, $lines);
            case '신한은행':
                return $this->parseShinhanBank($smsBody, $lines);
            case '국민은행':
            case 'KB':
                return $this->parseKBBank($smsBody, $lines);
            case '우리은행':
                return $this->parseWooriBank($smsBody, $lines);
            case '하나은행':
                return $this->parseHanaBank($smsBody, $lines);
            case '농협':
            case 'NH농협':
                return $this->parseNHBank($smsBody, $lines);
            case '토스뱅크':
                return $this->parseTossBank($smsBody, $lines);
            case 'DGB대구은행':
            case '대구은행':
                return $this->parseDGBBank($smsBody, $lines);
            case '케이뱅크':  // ✨ 추가!
                return $this->parseKBank($smsBody, $lines);
            default:
                return $this->parseGenericBank($smsBody, $lines, $bank);
        }
    }
    
    private function identifyBank($smsBody) {
        $banks = [
            '카카오뱅크' => ['[카카오뱅크]'],
            '신한은행' => ['[신한은행]'],
            'KB' => ['[KB]'],
            '국민은행' => ['[국민은행]', '[KB국민은행]'],
            '우리은행' => ['[우리은행]'],
            '하나은행' => ['[하나은행]', '하나,'],
            '농협' => ['[농협]', '[NH농협]', '농협 '],
            '기업은행' => ['[기업은행]', '[IBK기업은행]'],
            '새마을금고' => ['[새마을금고]'],
            '토스뱅크' => ['[토스뱅크]'],
            'DGB대구은행' => ['iM뱅크', 'DGB'],  // ✨ 추가
            '대구은행' => ['대구은행'],  // ✨ 추가
            '케이뱅크' => ['[케이뱅크]', '[K뱅크]']  // ✨ 추가!
        ];
        
        foreach ($banks as $bankName => $patterns) {
            foreach ($patterns as $pattern) {
                if (strpos($smsBody, $pattern) !== false) {
                    return $bankName;
                }
            }
        }
        
        return '알수없음';
    }
    
    private function parseKakaoBank($smsBody, $lines) {
        // [Web발신] [카카오뱅크] 이*해(4771) 05/22 00:07 입금 400,000원 김성진 잔액 2,435,000원
        preg_match('/([^(]+)\((\d+)\)/', $smsBody, $accountMatch);
        preg_match('/(입금|출금)\s*([\d,]+)원/', $smsBody, $transactionMatch);
        preg_match('/잔액\s*([\d,]+)원/', $smsBody, $balanceMatch);
        
        return [
            'bank' => '카카오뱅크',
            'accountHolder' => $accountMatch[1] ?? '알수없음',
            'accountNumber' => $accountMatch[2] ?? '알수없음',
            'transactionType' => $transactionMatch[1] ?? null,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }

    private function parseKBank($smsBody, $lines) {
        // [Web발신]
        // [케이뱅크]
        // 김민*(5086)
        // 입금 100,000원
        // 잔액 11,382,437원
        // 성우석
        
        preg_match('/([^(]+)\((\d+)\)/', $smsBody, $accountMatch);
        preg_match('/(입금|출금)\s*([\d,]+)원/', $smsBody, $transactionMatch);
        preg_match('/잔액\s*([\d,]+)원/', $smsBody, $balanceMatch);
        
        // 입금자명 추출
        $depositorName = '알수없음';
        $lines = explode("\n", trim($smsBody));
        
        // 마지막 줄에서 한글만 있는 줄 찾기
        for ($i = count($lines) - 1; $i >= 0; $i--) {
            $line = trim($lines[$i]);
            // 한글만 있고 2글자 이상이면 입금자명
            if (preg_match('/^[가-힣]+$/', $line) && mb_strlen($line) >= 2) {
                $depositorName = $line;
                break;
            }
        }
        
        return [
            'bank' => '케이뱅크',
            'accountHolder' => $accountMatch[1] ?? '알수없음',  // 김민*
            'accountNumber' => $accountMatch[2] ?? '알수없음',  // 5086
            'transactionType' => $transactionMatch[1] ?? null,  // 입금
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,  // 100000
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,  // 11382437
            'depositorName' => $depositorName,  // ✨ 성우석
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }
    
    private function parseKBBank($smsBody, $lines) {
        // [KB]05/22 00:12 759302**377 김종현 전자금융입금 4,000,000 잔액5,850,375
        preg_match('/(\d+\*+\d+)/', $smsBody, $accountMatch);
        preg_match('/([가-힣]+)\s+(전자금융입금|전자금융출금|입금|출금)/', $smsBody, $nameMatch);
        preg_match('/(전자금융입금|전자금융출금|입금|출금)\s*([\d,]+)/', $smsBody, $transactionMatch);
        preg_match('/잔액\s*([\d,]+)/', $smsBody, $balanceMatch);
        
        $transactionType = null;
        if (isset($transactionMatch[1])) {
            $transactionType = (strpos($transactionMatch[1], '입금') !== false) ? '입금' : '출금';
        }
        
        return [
            'bank' => 'KB',
            'accountHolder' => $nameMatch[1] ?? '알수없음',
            'accountNumber' => $accountMatch[1] ?? '알수없음',
            'transactionType' => $transactionType,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }
    
    private function parseHanaBank($smsBody, $lines) {
    // 기존: 하나,05/22,00:07 632******36207 입금50,000원 박효훈 잔액50,000원
    // 추가: [하나은행] 홍길동(7777) 입금 200,000원 잔액 1,175,000원 계좌이체
    
    preg_match('/(\d+\*+\d+)/', $smsBody, $accountMatch);
    
    // [하나은행] 형식 처리 추가
    if (strpos($smsBody, '[하나은행]') !== false) {
        // [하나은행] 홍길동(7777) 입금 200,000원 잔액 1,175,000원 계좌이체
        preg_match('/([가-힣]+)\((\d+)\)/', $smsBody, $nameAccountMatch);
        preg_match('/(입금|출금)\s*([\d,]+)원/', $smsBody, $transactionMatch);
        preg_match('/잔액\s*([\d,]+)원/', $smsBody, $balanceMatch);
        
        return [
            'bank' => '하나은행',
            'accountHolder' => $nameAccountMatch[1] ?? '알수없음',
            'accountNumber' => $nameAccountMatch[2] ?? '알수없음',
            'transactionType' => $transactionMatch[1] ?? null,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }
    
    // 기존 하나, 형식 처리
    preg_match('/(입금|출금)([\d,]+)원/', $smsBody, $transactionMatch);
    preg_match('/잔액\s*([\d,]+)원/', $smsBody, $balanceMatch);
    preg_match('/(입금|출금)[\d,]+원\s*([가-힣]+)\s*잔액/', $smsBody, $nameMatch);
    
    return [
        'bank' => '하나은행',
        'accountHolder' => $nameMatch[2] ?? '알수없음',
        'accountNumber' => $accountMatch[1] ?? '알수없음',
        'transactionType' => $transactionMatch[1] ?? null,
        'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
        'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
        'memo' => $this->extractMemo($smsBody),
        'timestamp' => date('Y-m-d H:i:s'),
        'originalText' => $smsBody
        ];
    }
    
    private function parseShinhanBank($smsBody, $lines) {
        // [신한은행] 12/25 14:30 김철수 1234 출금 50,000원 잔액 1,200,000원
        preg_match('/\d+:\d+\s+(\S+)/', $smsBody, $nameMatch);
        preg_match('/(\d{4})/', $smsBody, $accountMatch);
        preg_match('/(입금|출금)\s+([\d,]+)원/', $smsBody, $transactionMatch);
        preg_match('/잔액\s+([\d,]+)원/', $smsBody, $balanceMatch);
        
        return [
            'bank' => '신한은행',
            'accountHolder' => $nameMatch[1] ?? '알수없음',
            'accountNumber' => $accountMatch[1] ?? '알수없음',
            'transactionType' => $transactionMatch[1] ?? null,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }
    
    private function parseWooriBank($smsBody, $lines) {
        // [우리은행] 홍길동(1234) 12/25 출금 100,000원 잔액 900,000원 ATM
        preg_match('/(\S+)\((\d+)\)/', $smsBody, $accountMatch);
        preg_match('/(입금|출금)\s+([\d,]+)원/', $smsBody, $transactionMatch);
        preg_match('/잔액\s+([\d,]+)원/', $smsBody, $balanceMatch);
        
        return [
            'bank' => '우리은행',
            'accountHolder' => $accountMatch[1] ?? '알수없음',
            'accountNumber' => $accountMatch[2] ?? '알수없음',
            'transactionType' => $transactionMatch[1] ?? null,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }
    
    private function parseNHBank($smsBody, $lines) {
        // 농협 출금2,930,000원 05/22 00:16 356-****-6174-83 주식회사 직 잔액0원
        preg_match('/(\d+-\*+-\d+-\d+)/', $smsBody, $accountMatch);
        preg_match('/(입금|출금)([\d,]+)원/', $smsBody, $transactionMatch);
        preg_match('/잔액\s*([\d,]+)원/', $smsBody, $balanceMatch);
        
        return [
            'bank' => '농협',
            'accountHolder' => '알수없음',
            'accountNumber' => $accountMatch[1] ?? '알수없음',
            'transactionType' => $transactionMatch[1] ?? null,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }
    
    private function parseTossBank($smsBody, $lines) {
        // [토스뱅크] 홍길동 출금 25,000원 잔액 975,000원
        preg_match('/^(\S+)/', $lines[1] ?? '', $nameMatch);
        preg_match('/(입금|출금)\s+([\d,]+)원/', $smsBody, $transactionMatch);
        preg_match('/잔액\s+([\d,]+)원/', $smsBody, $balanceMatch);
        
        return [
            'bank' => '토스뱅크',
            'accountHolder' => $nameMatch[1] ?? '알수없음',
            'accountNumber' => '알수없음',
            'transactionType' => $transactionMatch[1] ?? null,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }
    
    private function parseGenericBank($smsBody, $lines, $bank) {
        preg_match('/(입금|출금)\s*([\d,]+)원/', $smsBody, $transactionMatch);
        preg_match('/잔액\s*([\d,]+)원/', $smsBody, $balanceMatch);
        
        return [
            'bank' => $bank,
            'accountHolder' => '알수없음',
            'accountNumber' => '알수없음',
            'transactionType' => $transactionMatch[1] ?? null,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }

    // 파싱 함수 추가
    private function parseDGBBank($smsBody, $lines) {
        // iM뱅크 10/24 02:40
        // 50815*47781
        // 입금 50,000
        // 잔액 7,175,650
        // 카뱅이성호
        
        preg_match('/(\d+\*+\d+)/', $smsBody, $accountMatch);
        preg_match('/(입금|출금)\s*([\d,]+)/', $smsBody, $transactionMatch);
        preg_match('/잔액\s*([\d,]+)/', $smsBody, $balanceMatch);
        preg_match('/(입금|출금)\s*[\d,]+\s+([가-힣a-zA-Z]+)/', $smsBody, $depositorMatch);

        // 입금자명 추출 ("카뱅이성호" → "이성호")
        $lines = explode("\n", trim($smsBody));
        $depositorName = '알수없음';
        foreach ($lines as $line) {
            $line = trim($line);
            // 한글만 있는 줄 찾기 (입금자명)
            if (preg_match('/^[가-힣]+$/', $line) && mb_strlen($line) >= 2) {
                $depositorName = $line;
                // "카뱅이성호" 같은 경우
                if (mb_strlen($depositorName) > 3 && mb_substr($depositorName, 0, 2) === '카뱅') {
                    $depositorName = mb_substr($depositorName, 2);
                }
                break;
            }
        }
        
        return [
            'bank' => 'DGB대구은행',
            'accountHolder' => '알수없음',
            'accountNumber' => $accountMatch[1] ?? '알수없음',
            'transactionType' => $transactionMatch[1] ?? null,
            'amount' => isset($transactionMatch[2]) ? (int)str_replace(',', '', $transactionMatch[2]) : 0,
            'balance' => isset($balanceMatch[1]) ? (int)str_replace(',', '', $balanceMatch[1]) : 0,
            'depositorName' => $depositorName,  // ✨ 입금자명
            'memo' => $this->extractMemo($smsBody),
            'timestamp' => date('Y-m-d H:i:s'),
            'originalText' => $smsBody
        ];
    }
    
    private function extractMemo($smsBody) {
        $memos = [];
        if (strpos($smsBody, 'ATM') !== false) $memos[] = 'ATM';
        if (strpos($smsBody, '체크카드') !== false) $memos[] = '체크카드';
        if (strpos($smsBody, '계좌이체') !== false) $memos[] = '계좌이체';
        if (strpos($smsBody, '전자금융') !== false) $memos[] = '전자금융';
        
        return implode(', ', $memos);
    }
}

?>