<?php
include 'layout/header.php';
?>

<style>
.setting-card {
  background: rgba(255, 255, 255, 0.05);
  border: 1px solid rgba(255, 255, 255, 0.1);
  border-radius: 12px;
  padding: 24px;
  margin-bottom: 20px;
}

.toggle-switch {
  position: relative;
  display: inline-block;
  width: 60px;
  height: 34px;
}

.toggle-switch input {
  opacity: 0;
  width: 0;
  height: 0;
}

.slider {
  position: absolute;
  cursor: pointer;
  top: 0;
  left: 0;
  right: 0;
  bottom: 0;
  background-color: #ccc;
  transition: .4s;
  border-radius: 34px;
}

.slider:before {
  position: absolute;
  content: "";
  height: 26px;
  width: 26px;
  left: 4px;
  bottom: 4px;
  background-color: white;
  transition: .4s;
  border-radius: 50%;
}

input:checked + .slider {
  background-color: #10b981;
}

input:checked + .slider:before {
  transform: translateX(26px);
}

.volume-slider {
  width: 100%;
  -webkit-appearance: none;
  height: 6px;
  border-radius: 5px;
  background: #4a5568;
  outline: none;
  opacity: 0.7;
  transition: opacity 0.2s;
}

.volume-slider:hover {
  opacity: 1;
}

.volume-slider::-webkit-slider-thumb {
  -webkit-appearance: none;
  appearance: none;
  width: 20px;
  height: 20px;
  border-radius: 50%;
  background: #10b981;
  cursor: pointer;
}

.volume-slider::-moz-range-thumb {
  width: 20px;
  height: 20px;
  border-radius: 50%;
  background: #10b981;
  cursor: pointer;
}

.sound-item {
  background: rgba(255, 255, 255, 0.03);
  border: 1px solid rgba(255, 255, 255, 0.05);
  border-radius: 8px;
  padding: 16px;
  margin-bottom: 12px;
}

.play-button {
  width: 40px;
  height: 40px;
  border-radius: 50%;
  background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  transition: transform 0.2s;
}

.play-button:hover {
  transform: scale(1.1);
}

.sound-upload-area {
  border: 2px dashed rgba(255, 255, 255, 0.2);
  border-radius: 8px;
  padding: 20px;
  text-align: center;
  cursor: pointer;
  transition: all 0.3s;
}

.sound-upload-area:hover {
  border-color: rgba(255, 255, 255, 0.4);
  background: rgba(255, 255, 255, 0.05);
}

.sound-upload-area.dragover {
  border-color: #10b981;
  background: rgba(16, 185, 129, 0.1);
}

.status-indicator {
  display: inline-flex;
  align-items: center;
  padding: 0.25rem 0.75rem;
  border-radius: 9999px;
  font-size: 0.75rem;
  font-weight: 500;
}

.status-active {
  background-color: rgba(34, 197, 94, 0.2);
  color: rgb(34, 197, 94);
}

.status-inactive {
  background-color: rgba(107, 114, 128, 0.2);
  color: rgb(107, 114, 128);
}

.alert-message {
  position: fixed;
  top: 20px;
  right: 20px;
  padding: 15px 20px;
  border-radius: 8px;
  color: white;
  font-weight: 500;
  z-index: 9999;
  animation: slideIn 0.3s ease-out;
  box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.alert-success {
  background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}

.alert-error {
  background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
}

@keyframes slideIn {
  from {
    transform: translateX(100%);
    opacity: 0;
  }
  to {
    transform: translateX(0);
    opacity: 1;
  }
}

@keyframes slideOut {
  from {
    transform: translateX(0);
    opacity: 1;
  }
  to {
    transform: translateX(100%);
    opacity: 0;
  }
}
</style>

<main class="flex-1 p-8 space-y-8">
  <div class="flex justify-between items-center">
    <h2 class="text-3xl font-bold text-primary mb-6">사운드 설정</h2>
    <button id="saveButton" onclick="saveAllSettings()" class="px-6 py-2 bg-primary text-white rounded-lg hover:bg-teal-600 transition">
      설정 저장
    </button>
  </div>

  <!-- 전체 사운드 설정 -->
  <div class="setting-card">
    <div class="flex items-center justify-between mb-6">
      <div>
        <h3 class="text-xl font-semibold text-white mb-2">전체 알림음</h3>
        <p class="text-sm text-gray-400">모든 알림음을 켜거나 끌 수 있습니다</p>
      </div>
      <label class="toggle-switch">
        <input type="checkbox" id="master-sound-enabled" checked onchange="toggleMasterSound()">
        <span class="slider"></span>
      </label>
    </div>

    <div class="flex items-center gap-4">
      <span class="text-gray-400">음량</span>
      <input type="range" id="master-volume" class="volume-slider flex-1" min="0" max="100" value="70">
      <span id="master-volume-value" class="text-white font-semibold">70%</span>
    </div>
  </div>

  <!-- 출금 알림음 설정 -->
  <div class="setting-card">
    <div class="flex items-center justify-between mb-6">
      <div>
        <h3 class="text-xl font-semibold text-white mb-2">출금 알림음</h3>
        <p class="text-sm text-gray-400">출금 신청 시 재생되는 알림음입니다</p>
      </div>
      <label class="toggle-switch">
        <input type="checkbox" id="withdrawal-sound-enabled" checked onchange="toggleSound('withdrawal')">
        <span class="slider"></span>
      </label>
    </div>

    <div class="sound-item">
      <div class="flex items-center justify-between">
        <div class="flex items-center gap-4">
          <div class="play-button" onclick="playSound('withdrawal')">
            <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z"></path>
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
          </div>
          <div>
            <div id="withdrawal-filename" class="text-white font-medium">exchange.mp3</div>
            <div class="text-xs text-gray-500">현재 사용 중인 파일</div>
          </div>
        </div>
        <button onclick="uploadNewSound('withdrawal')" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition text-sm">
          파일 변경
        </button>
      </div>
    </div>

    <div class="flex items-center gap-4 mt-4">
      <span class="text-gray-400">음량</span>
      <input type="range" id="withdrawal-volume" class="volume-slider flex-1" min="0" max="100" value="70">
      <span id="withdrawal-volume-value" class="text-white font-semibold">70%</span>
    </div>
  </div>

  <!-- 입금 대기 알림음 설정 -->
  <div class="setting-card">
    <div class="flex items-center justify-between mb-6">
      <div>
        <h3 class="text-xl font-semibold text-white mb-2">입금 요청(신청) 알림음</h3>
        <p class="text-sm text-gray-400">입금 신청(Pending) 시 재생되는 알림음입니다</p>
      </div>
      <label class="toggle-switch">
        <input type="checkbox" id="deposit-pending-sound-enabled" checked onchange="toggleSound('deposit-pending')">
        <span class="slider"></span>
      </label>
    </div>

    <div class="sound-item">
      <div class="flex items-center justify-between">
        <div class="flex items-center gap-4">
          <div class="play-button" onclick="playSound('deposit-pending')">
            <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z"></path>
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
          </div>
          <div>
            <div id="deposit-pending-filename" class="text-white font-medium">charge.mp3</div>
            <div class="text-xs text-gray-500">현재 사용 중인 파일</div>
          </div>
        </div>
        <button onclick="uploadNewSound('deposit-pending')" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition text-sm">
          파일 변경
        </button>
      </div>
    </div>

    <div class="flex items-center gap-4 mt-4">
      <span class="text-gray-400">음량</span>
      <input type="range" id="deposit-pending-volume" class="volume-slider flex-1" min="0" max="100" value="50">
      <span id="deposit-pending-volume-value" class="text-white font-semibold">50%</span>
    </div>

    <!-- Pending 반복 재생 설정 추가 -->
    <div class="mt-4 p-4 bg-blue-500/10 border border-blue-500/20 rounded-lg">
      <div class="flex items-center justify-between">
        <div>
          <h4 class="text-blue-300 font-semibold">반복 재생</h4>
          <p class="text-blue-200 text-sm mt-1">대기 중인 입금이 있을 때 10초마다 반복 재생</p>
        </div>
        <label class="toggle-switch">
          <input type="checkbox" id="deposit-pending-repeat" checked>
          <span class="slider"></span>
        </label>
      </div>
    </div>
  </div>

  <!-- 입금 요청 알림음 설정 -->
  <div class="setting-card">
    <div class="flex items-center justify-between mb-6">
      <div>
        <h3 class="text-xl font-semibold text-white mb-2">입금 대기(송금완료) 알림음</h3>
        <p class="text-sm text-gray-400">입금 대기(Requested) 시 재생되는 알림음입니다</p>
      </div>
      <label class="toggle-switch">
        <input type="checkbox" id="deposit-requested-sound-enabled" checked onchange="toggleSound('deposit-requested')">
        <span class="slider"></span>
      </label>
    </div>

    <div class="sound-item">
      <div class="flex items-center justify-between">
        <div class="flex items-center gap-4">
          <div class="play-button" onclick="playSound('deposit-requested')">
            <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z"></path>
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
          </div>
          <div>
            <div id="deposit-requested-filename" class="text-white font-medium">request.mp3</div>
            <div class="text-xs text-gray-500">현재 사용 중인 파일</div>
          </div>
        </div>
        <button onclick="uploadNewSound('deposit-requested')" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition text-sm">
          파일 변경
        </button>
      </div>
    </div>

    <div class="flex items-center gap-4 mt-4">
      <span class="text-gray-400">음량</span>
      <input type="range" id="deposit-requested-volume" class="volume-slider flex-1" min="0" max="100" value="60">
      <span id="deposit-requested-volume-value" class="text-white font-semibold">60%</span>
    </div>

    <!-- 반복 재생 설정 -->
    <div class="mt-4 p-4 bg-yellow-500/10 border border-yellow-500/20 rounded-lg">
      <div class="flex items-center justify-between">
        <div>
          <h4 class="text-yellow-300 font-semibold">반복 재생</h4>
          <p class="text-yellow-200 text-sm mt-1">미확인 요청이 있을 때 10초마다 반복 재생</p>
        </div>
        <label class="toggle-switch">
          <input type="checkbox" id="deposit-requested-repeat" checked>
          <span class="slider"></span>
        </label>
      </div>
    </div>
  </div>

  <!-- 파일 업로드 영역 (숨김) -->
  <input type="file" id="sound-file-input" accept="audio/*" style="display: none;" onchange="handleFileUpload(event)">

  <!-- 테스트 영역 -->
  <div class="setting-card">
    <h3 class="text-xl font-semibold text-white mb-4">알림음 테스트</h3>
    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
      <button onclick="testSound('withdrawal')" class="px-4 py-3 bg-gradient-to-r from-purple-600 to-pink-600 text-white rounded-lg hover:opacity-90 transition">
        출금 알림 테스트
      </button>
      <button onclick="testSound('deposit-pending')" class="px-4 py-3 bg-gradient-to-r from-blue-600 to-cyan-600 text-white rounded-lg hover:opacity-90 transition">
        입금 대기 테스트
      </button>
      <button onclick="testSound('deposit-requested')" class="px-4 py-3 bg-gradient-to-r from-orange-600 to-red-600 text-white rounded-lg hover:opacity-90 transition">
        입금 요청 테스트
      </button>
    </div>
  </div>

  <!-- 설정 상태 표시 -->
  <div class="bg-white/5 border border-white/10 rounded-xl p-6">
    <h3 class="text-lg font-semibold text-white mb-4">현재 설정 상태</h3>
    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
      <div class="flex items-center justify-between">
        <span class="text-gray-300">전체 알림음</span>
        <span id="master-status" class="status-indicator status-active">활성화</span>
      </div>
      <div class="flex items-center justify-between">
        <span class="text-gray-300">출금 알림</span>
        <span id="withdrawal-status" class="status-indicator status-active">활성화</span>
      </div>
      <div class="flex items-center justify-between">
        <span class="text-gray-300">입금 대기 알림</span>
        <span id="deposit-pending-status" class="status-indicator status-active">활성화</span>
      </div>
      <div class="flex items-center justify-between">
        <span class="text-gray-300">입금 요청 알림</span>
        <span id="deposit-requested-status" class="status-indicator status-active">활성화</span>
      </div>
    </div>
  </div>
</main>

<script>
// 현재 업로드 대상
let currentUploadTarget = null;

// 알림 메시지 표시
function showAlert(message, isSuccess = true) {
    // 기존 알림 제거
    const existingAlert = document.querySelector('.alert-message');
    if (existingAlert) {
        existingAlert.remove();
    }
    
    // 새 알림 생성
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert-message ${isSuccess ? 'alert-success' : 'alert-error'}`;
    alertDiv.textContent = message;
    document.body.appendChild(alertDiv);
    
    // 3초 후 제거
    setTimeout(() => {
        alertDiv.style.animation = 'slideOut 0.3s ease-out';
        setTimeout(() => {
            alertDiv.remove();
        }, 300);
    }, 3000);
}

// 볼륨 슬라이더 이벤트
document.getElementById('master-volume').addEventListener('input', function() {
    document.getElementById('master-volume-value').textContent = this.value + '%';
});

document.getElementById('withdrawal-volume').addEventListener('input', function() {
    document.getElementById('withdrawal-volume-value').textContent = this.value + '%';
});

document.getElementById('deposit-pending-volume').addEventListener('input', function() {
    document.getElementById('deposit-pending-volume-value').textContent = this.value + '%';
});

document.getElementById('deposit-requested-volume').addEventListener('input', function() {
    document.getElementById('deposit-requested-volume-value').textContent = this.value + '%';
});

// 페이지 로드 시 현재 설정 가져오기
document.addEventListener('DOMContentLoaded', function() {
    loadCurrentSettings();
});

// 현재 설정 가져오기
function loadCurrentSettings() {
    fetch('ajax/get_sound_settings.php')
        .then(response => {
            if (!response.ok) {
                throw new Error('네트워크 응답 오류');
            }
            return response.json();
        })
        .then(data => {
            console.log('로드된 설정:', data);
            
            if (data.success && data.settings) {
                const settings = data.settings;
                
                // 전체 사운드 설정
                document.getElementById('master-sound-enabled').checked = settings.sound_enabled == 1;
                document.getElementById('master-volume').value = settings.master_volume || 70;
                document.getElementById('master-volume-value').textContent = (settings.master_volume || 70) + '%';
                
                // 출금 알림
                document.getElementById('withdrawal-sound-enabled').checked = settings.withdrawal_sound_enabled == 1;
                document.getElementById('withdrawal-volume').value = settings.withdrawal_volume || 70;
                document.getElementById('withdrawal-volume-value').textContent = (settings.withdrawal_volume || 70) + '%';
                
                // 파일명 업데이트
                if (settings.withdrawal_sound_file) {
                    const filename = settings.withdrawal_sound_file.split('/').pop();
                    document.getElementById('withdrawal-filename').textContent = filename;
                }
                
                // 입금 대기 알림
                document.getElementById('deposit-pending-sound-enabled').checked = settings.deposit_pending_sound_enabled == 1;
                document.getElementById('deposit-pending-volume').value = settings.deposit_pending_volume || 50;
                document.getElementById('deposit-pending-volume-value').textContent = (settings.deposit_pending_volume || 50) + '%';
                
                if (settings.deposit_pending_sound_file) {
                    const filename = settings.deposit_pending_sound_file.split('/').pop();
                    document.getElementById('deposit-pending-filename').textContent = filename;
                }
                
                // Pending 반복 설정
                const pendingRepeatEl = document.getElementById('deposit-pending-repeat');
                if (pendingRepeatEl) {
                    if (settings.deposit_pending_repeat !== undefined) {
                        pendingRepeatEl.checked = settings.deposit_pending_repeat == 1;
                    } else {
                        pendingRepeatEl.checked = true;
                    }
                }
                
                // 입금 요청 알림
                document.getElementById('deposit-requested-sound-enabled').checked = settings.deposit_requested_sound_enabled == 1;
                document.getElementById('deposit-requested-volume').value = settings.deposit_requested_volume || 60;
                document.getElementById('deposit-requested-volume-value').textContent = (settings.deposit_requested_volume || 60) + '%';
                document.getElementById('deposit-requested-repeat').checked = settings.deposit_requested_repeat == 1;
                
                if (settings.deposit_requested_sound_file) {
                    const filename = settings.deposit_requested_sound_file.split('/').pop();
                    document.getElementById('deposit-requested-filename').textContent = filename;
                }
                
                // 마스터가 꺼져있으면 다른 설정들도 비활성화
                if (settings.sound_enabled != 1) {
                    toggleMasterSound();
                }
                
                updateStatusIndicators();
            }
        })
        .catch(error => {
            console.error('설정 로드 실패:', error);
            showAlert('설정을 불러오는데 실패했습니다.', false);
        });
}

// 마스터 사운드 토글
function toggleMasterSound() {
    const enabled = document.getElementById('master-sound-enabled').checked;
    
    // 모든 개별 사운드 설정 비활성화/활성화
    const soundControls = ['withdrawal', 'deposit-pending', 'deposit-requested'];
    soundControls.forEach(control => {
        const element = document.getElementById(control + '-sound-enabled');
        element.disabled = !enabled;
        
        // 시각적 피드백
        const card = element.closest('.setting-card');
        if (!enabled) {
            card.style.opacity = '0.5';
        } else {
            card.style.opacity = '1';
        }
    });
    
    updateStatusIndicators();
}

// 개별 사운드 토글
function toggleSound(type) {
    updateStatusIndicators();
}

// 상태 표시기 업데이트
function updateStatusIndicators() {
    // 마스터 상태
    const masterEnabled = document.getElementById('master-sound-enabled').checked;
    const masterStatus = document.getElementById('master-status');
    
    if (masterEnabled) {
        masterStatus.textContent = '활성화';
        masterStatus.className = 'status-indicator status-active';
    } else {
        masterStatus.textContent = '비활성화';
        masterStatus.className = 'status-indicator status-inactive';
    }
    
    // 출금 상태
    const withdrawalEnabled = document.getElementById('withdrawal-sound-enabled').checked;
    const withdrawalStatus = document.getElementById('withdrawal-status');
    
    if (withdrawalEnabled && masterEnabled) {
        withdrawalStatus.textContent = '활성화';
        withdrawalStatus.className = 'status-indicator status-active';
    } else {
        withdrawalStatus.textContent = '비활성화';
        withdrawalStatus.className = 'status-indicator status-inactive';
    }
    
    // 입금 대기 상태
    const pendingEnabled = document.getElementById('deposit-pending-sound-enabled').checked;
    const pendingStatus = document.getElementById('deposit-pending-status');
    
    if (pendingEnabled && masterEnabled) {
        pendingStatus.textContent = '활성화';
        pendingStatus.className = 'status-indicator status-active';
    } else {
        pendingStatus.textContent = '비활성화';
        pendingStatus.className = 'status-indicator status-inactive';
    }
    
    // 입금 요청 상태
    const requestedEnabled = document.getElementById('deposit-requested-sound-enabled').checked;
    const requestedStatus = document.getElementById('deposit-requested-status');
    
    if (requestedEnabled && masterEnabled) {
        requestedStatus.textContent = '활성화';
        requestedStatus.className = 'status-indicator status-active';
    } else {
        requestedStatus.textContent = '비활성화';
        requestedStatus.className = 'status-indicator status-inactive';
    }
}

// 사운드 재생
function playSound(type) {
    let soundFile = '';
    let volume = 0.7;
    
    // 현재 설정된 파일명 가져오기
    switch(type) {
        case 'withdrawal':
            const withdrawalFile = document.getElementById('withdrawal-filename').textContent;
            soundFile = withdrawalFile.includes('/') ? withdrawalFile : '/assets/sound/' + withdrawalFile;
            volume = document.getElementById('withdrawal-volume').value / 100;
            break;
        case 'deposit-pending':
            const pendingFile = document.getElementById('deposit-pending-filename').textContent;
            soundFile = pendingFile.includes('/') ? pendingFile : '/assets/sound/' + pendingFile;
            volume = document.getElementById('deposit-pending-volume').value / 100;
            break;
        case 'deposit-requested':
            const requestedFile = document.getElementById('deposit-requested-filename').textContent;
            soundFile = requestedFile.includes('/') ? requestedFile : '/assets/sound/' + requestedFile;
            volume = document.getElementById('deposit-requested-volume').value / 100;
            break;
    }
    
    const masterVolume = document.getElementById('master-volume').value / 100;
    const audio = new Audio(soundFile);
    audio.volume = volume * masterVolume;
    audio.play().catch(error => {
        console.error('사운드 재생 오류:', error);
        showAlert('사운드 재생에 실패했습니다.', false);
    });
}

// 테스트 사운드 재생
function testSound(type) {
    const masterEnabled = document.getElementById('master-sound-enabled').checked;
    const soundEnabled = document.getElementById(type + '-sound-enabled').checked;
    
    if (!masterEnabled) {
        showAlert('전체 알림음이 꺼져 있습니다.', false);
        return;
    }
    
    if (!soundEnabled) {
        showAlert('해당 알림음이 꺼져 있습니다.', false);
        return;
    }
    
    playSound(type);
    showAlert('테스트 사운드를 재생합니다.', true);
}

// 파일 업로드
function uploadNewSound(type) {
    currentUploadTarget = type;
    document.getElementById('sound-file-input').click();
}

// 파일 업로드 처리
function handleFileUpload(event) {
    const file = event.target.files[0];
    if (!file) return;
    
    // 파일 크기 체크 (5MB 제한)
    if (file.size > 5 * 1024 * 1024) {
        showAlert('파일 크기는 5MB 이하여야 합니다.', false);
        return;
    }
    
    // 오디오 파일 체크
    if (!file.type.startsWith('audio/')) {
        showAlert('오디오 파일만 업로드 가능합니다.', false);
        return;
    }
    
    const formData = new FormData();
    formData.append('sound_file', file);
    formData.append('type', currentUploadTarget);
    
    // 업로드 중 표시
    showAlert('파일을 업로드하고 있습니다...', true);
    
    fetch('ajax/upload_sound.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('사운드 파일이 성공적으로 변경되었습니다.', true);
            
            // 파일명 업데이트
            const filename = data.filename || file.name;
            switch(currentUploadTarget) {
                case 'withdrawal':
                    document.getElementById('withdrawal-filename').textContent = filename;
                    break;
                case 'deposit-pending':
                    document.getElementById('deposit-pending-filename').textContent = filename;
                    break;
                case 'deposit-requested':
                    document.getElementById('deposit-requested-filename').textContent = filename;
                    break;
            }
        } else {
            showAlert(data.message || '파일 업로드 중 오류가 발생했습니다.', false);
        }
    })
    .catch(error => {
        console.error('업로드 오류:', error);
        showAlert('파일 업로드 중 오류가 발생했습니다.', false);
    });
    
    // 입력 초기화
    event.target.value = '';
}

// 모든 설정 저장
function saveAllSettings() {
    const saveButton = document.getElementById('saveButton');
    saveButton.disabled = true;
    saveButton.textContent = '저장 중...';
    
    const settingsData = {
        sound_enabled: document.getElementById('master-sound-enabled').checked ? 1 : 0,
        master_volume: parseInt(document.getElementById('master-volume').value),
        withdrawal_sound_enabled: document.getElementById('withdrawal-sound-enabled').checked ? 1 : 0,
        withdrawal_volume: parseInt(document.getElementById('withdrawal-volume').value),
        deposit_pending_sound_enabled: document.getElementById('deposit-pending-sound-enabled').checked ? 1 : 0,
        deposit_pending_volume: parseInt(document.getElementById('deposit-pending-volume').value),
        deposit_requested_sound_enabled: document.getElementById('deposit-requested-sound-enabled').checked ? 1 : 0,
        deposit_requested_volume: parseInt(document.getElementById('deposit-requested-volume').value),
        deposit_requested_repeat: document.getElementById('deposit-requested-repeat').checked ? 1 : 0
    };
    
    // Pending 반복 설정 추가
    const pendingRepeatEl = document.getElementById('deposit-pending-repeat');
    if (pendingRepeatEl) {
        settingsData.deposit_pending_repeat = pendingRepeatEl.checked ? 1 : 0;
    }
    
    console.log('저장할 데이터:', settingsData);
    
    fetch('ajax/save_sound_settings.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(settingsData)
    })
    .then(response => {
        console.log('응답 상태:', response.status);
        if (!response.ok) {
            throw new Error('네트워크 응답 오류');
        }
        return response.json();
    })
    .then(data => {
        console.log('응답 데이터:', data);
        saveButton.disabled = false;
        saveButton.textContent = '설정 저장';
        
        if (data.success) {
            showAlert('사운드 설정이 저장되었습니다.', true);
            updateStatusIndicators();
        } else {
            showAlert(data.message || '설정 저장에 실패했습니다.', false);
        }
    })
    .catch(error => {
        console.error('설정 저장 실패:', error);
        saveButton.disabled = false;
        saveButton.textContent = '설정 저장';
        showAlert('설정 저장 중 오류가 발생했습니다.', false);
    });
}
</script>

</body>
</html>