<?php
// update_transfer_status.php - 송금 상태 업데이트 (중복 처리 방지 추가)
session_start();
header('Content-Type: application/json; charset=utf-8');

try {
    include 'inc/db_connect.php';
    
    // POST 요청 확인
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('POST 요청만 허용됩니다.');
    }
    
    // JSON 데이터 읽기
    $json_input = file_get_contents('php://input');
    error_log("Transfer status update input: " . $json_input);
    
    if (empty($json_input)) {
        throw new Exception('요청 데이터가 비어있습니다.');
    }
    
    $input = json_decode($json_input, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('JSON 파싱 에러: ' . json_last_error_msg());
    }
    
    $transactionId = $input['transaction_id'] ?? '';
    $newStatus = $input['status'] ?? 'requested';
    
    if (!$transactionId) {
        throw new Exception('거래 ID가 필요합니다.');
    }
    
    // === 중복 처리 방지 로직 시작 ===
    
    // 1. 요청 고유 식별자 생성 (ip_address 제거)
    $request_key = $transactionId . '_' . $newStatus . '_' . 'user_request';
    $request_hash = md5($request_key);
    
    // 2. 처리 중인 요청인지 확인 (동시 요청 방지)
    if (!isset($_SESSION['processing_updates'])) {
        $_SESSION['processing_updates'] = [];
    }
    
    if (in_array($request_hash, $_SESSION['processing_updates'])) {
        echo json_encode([
            'success' => false,
            'message' => '현재 처리 중인 요청입니다. 잠시 후 다시 시도해주세요.',
            'code' => 'PROCESSING'
        ]);
        exit;
    }
    
    // 3. 최근 처리된 요청인지 확인 (중복 방지)
    if (!isset($_SESSION['recent_updates'])) {
        $_SESSION['recent_updates'] = [];
    }
    
    $current_time = time();
    
    // 최근 30초 이내 같은 요청이 있었는지 확인
    foreach ($_SESSION['recent_updates'] as $key => $update_info) {
        if ($current_time - $update_info['time'] > 30) {
            // 30초 지난 항목 제거
            unset($_SESSION['recent_updates'][$key]);
        } elseif ($update_info['hash'] === $request_hash) {
            echo json_encode([
                'success' => false,
                'message' => '이미 처리된 요청입니다.',
                'code' => 'DUPLICATE',
                'processed_at' => date('Y-m-d H:i:s', $update_info['time'])
            ]);
            exit;
        }
    }
    
    // 4. 처리 중 목록에 추가
    $_SESSION['processing_updates'][] = $request_hash;
    
    // === 중복 처리 방지 로직 끝 ===
    
    // 현재 거래 정보 조회
    $selectQuery = "SELECT id, status, payment_status FROM transfer WHERE kakao_tid = ?";
    $selectStmt = mysqli_prepare($con, $selectQuery);
    mysqli_stmt_bind_param($selectStmt, 's', $transactionId);
    mysqli_stmt_execute($selectStmt);
    $result = mysqli_stmt_get_result($selectStmt);
    $transfer = mysqli_fetch_assoc($result);
    
    if (!$transfer) {
        // 처리 중 목록에서 제거
        $key = array_search($request_hash, $_SESSION['processing_updates']);
        if ($key !== false) {
            unset($_SESSION['processing_updates'][$key]);
        }
        throw new Exception('해당 거래를 찾을 수 없습니다.');
    }
    
    $oldStatus = $transfer['status'];
    $transferId = $transfer['id'];
    
    // 상태 변경 가능 여부 확인
    $allowedTransitions = [
        'pending' => ['requested', 'cancelled'],
        'requested' => ['approved', 'rejected', 'cancelled'],
        'approved' => ['completed'],
        'rejected' => ['pending'], // 재신청 가능
        'cancelled' => ['pending'], // 재신청 가능
    ];
    
    if (!isset($allowedTransitions[$oldStatus]) || 
        !in_array($newStatus, $allowedTransitions[$oldStatus])) {
        
        // 처리 중 목록에서 제거
        $key = array_search($request_hash, $_SESSION['processing_updates']);
        if ($key !== false) {
            unset($_SESSION['processing_updates'][$key]);
        }
        
        // 이미 같은 상태인 경우 (중복 처리)
        if ($oldStatus === $newStatus) {
            echo json_encode([
                'success' => false,
                'message' => '이미 ' . $newStatus . ' 상태입니다.',
                'code' => 'SAME_STATUS',
                'current_status' => $oldStatus
            ]);
            exit;
        }
        
        throw new Exception(
            '상태 변경이 불가능합니다. ' . 
            '현재: ' . $oldStatus . ' → 요청: ' . $newStatus
        );
    }
    
    // 트랜잭션 시작
    mysqli_autocommit($con, false);
    
    try {
        // 상태 업데이트
        $updateQuery = "UPDATE transfer SET status = ?, updated_at = NOW() WHERE kakao_tid = ? AND status = ?";
        $updateStmt = mysqli_prepare($con, $updateQuery);
        mysqli_stmt_bind_param($updateStmt, 'sss', $newStatus, $transactionId, $oldStatus);
        $updateResult = mysqli_stmt_execute($updateStmt);
        
        if (!$updateResult) {
            throw new Exception('상태 업데이트에 실패했습니다: ' . mysqli_error($con));
        }
        
        $affectedRows = mysqli_affected_rows($con);
        
        if ($affectedRows === 0) {
            // 다른 프로세스에서 이미 상태를 변경한 경우
            throw new Exception('거래 상태가 이미 변경되었습니다. 페이지를 새로고침해주세요.');
        }
        
        // 상태 변경 로그 저장 (스키마에 맞게 수정)
        try {
            // transfer_status_logs 테이블의 changed_by는 DATETIME 타입이므로 제거
            $logQuery = "INSERT INTO transfer_status_logs (transfer_id, old_status, new_status, created_at) VALUES (?, ?, ?, NOW())";
            $logStmt = mysqli_prepare($con, $logQuery);
            
            if ($logStmt) {
                mysqli_stmt_bind_param($logStmt, 'iss', $transferId, $oldStatus, $newStatus);
                mysqli_stmt_execute($logStmt);
            }
        } catch (Exception $logError) {
            // 로그 저장 실패는 메인 기능에 영향을 주지 않도록 처리
            error_log("로그 저장 실패 (무시됨): " . $logError->getMessage());
        }
        
        // 트랜잭션 커밋
        mysqli_commit($con);
        
        // 처리 완료 후 세션 업데이트
        $_SESSION['recent_updates'][] = [
            'hash' => $request_hash,
            'time' => $current_time,
            'transaction_id' => $transactionId,
            'old_status' => $oldStatus,
            'new_status' => $newStatus
        ];
        
        // 처리 중 목록에서 제거
        $key = array_search($request_hash, $_SESSION['processing_updates']);
        if ($key !== false) {
            unset($_SESSION['processing_updates'][$key]);
            $_SESSION['processing_updates'] = array_values($_SESSION['processing_updates']);
        }
        
        // 성공 로그
        error_log("송금 상태 업데이트 완료: Transaction ID={$transactionId}, Status: {$oldStatus} → {$newStatus}");
        
        echo json_encode([
            'success' => true,
            'message' => '송금 상태가 업데이트되었습니다.',
            'data' => [
                'transaction_id' => $transactionId,
                'transfer_id' => $transferId,
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
                'updated_at' => date('Y-m-d H:i:s')
            ]
        ]);
        
    } catch (Exception $e) {
        // 트랜잭션 롤백
        mysqli_rollback($con);
        
        // 처리 중 목록에서 제거
        $key = array_search($request_hash, $_SESSION['processing_updates']);
        if ($key !== false) {
            unset($_SESSION['processing_updates'][$key]);
        }
        
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("송금 상태 업데이트 오류: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

// 세션 정리 (메모리 관리)
if (isset($_SESSION['recent_updates']) && count($_SESSION['recent_updates']) > 100) {
    $_SESSION['recent_updates'] = array_slice($_SESSION['recent_updates'], -50);
}

if (isset($_SESSION['processing_updates']) && count($_SESSION['processing_updates']) > 20) {
    $_SESSION['processing_updates'] = array_slice($_SESSION['processing_updates'], -10);
}
?>