<?php
include 'layout/header.php';

// 파트너 권한 체크
$partner_id = $_SESSION['id'];
?>

<!-- 모바일 메뉴 열기 버튼 -->
<button class="md:hidden fixed top-4 left-4 z-40 bg-gradient-to-r from-blue-600 to-purple-600 text-white p-3 rounded-full shadow-lg hover:shadow-xl transition-all duration-300" onclick="toggleSidebar()">
    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
    </svg>
</button>

<!-- 출금 신청 모달 -->
<div id="withdrawalModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
    <div class="bg-gradient-to-br from-gray-800 to-gray-900 p-8 rounded-2xl shadow-2xl border border-gray-700/50 backdrop-blur-sm w-full max-w-md max-h-[90vh] overflow-y-auto">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-2xl font-bold text-white">출금 신청</h2>
            <button onclick="closeWithdrawalModal()" class="text-gray-400 hover:text-white transition-colors">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>

        <!-- 잔액 정보 -->
        <div class="bg-gradient-to-r from-blue-600/10 to-purple-600/10 border border-blue-500/20 rounded-xl p-4 mb-6">
            <div class="flex items-center justify-between">
                <div>
                    <h3 class="text-sm font-medium text-gray-400 mb-1">출금 가능 금액</h3>
                    <p class="text-xl font-bold text-white" id="availableAmount">0원</p>
                </div>
                <div class="text-blue-400">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z"></path>
                    </svg>
                </div>
            </div>
            <div class="mt-3 p-3 bg-yellow-500/10 border border-yellow-500/20 rounded-lg">
                <p class="text-xs text-yellow-400 flex items-center">
                    <svg class="w-3 h-3 mr-2 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.664-.833-2.464 0L3.34 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                    </svg>
                    출금 시 수수료 1,000원이 차감됩니다.
                </p>
            </div>
        </div>

        <!-- 계좌 정보 -->
        <div id="bankInfo" class="bg-gray-700/50 rounded-xl p-4 mb-6 border border-gray-600/30">
            <div class="animate-pulse">
                <div class="flex items-center mb-2">
                    <div class="w-4 h-4 bg-gray-600 rounded mr-2"></div>
                    <div class="h-3 bg-gray-600 rounded w-20"></div>
                </div>
                <p class="text-gray-400 text-sm">계좌 정보를 불러오는 중...</p>
            </div>
        </div>
        
        <!-- 출금 신청 폼 -->
        <form id="withdrawalForm" class="space-y-4">
            <div class="space-y-2">
                <label for="withdrawAmount" class="block text-sm font-medium text-gray-300">출금 금액</label>
                <div class="relative">
                    <input type="number" 
                        name="amount" 
                        id="withdrawAmount"
                        placeholder="최소 10,000원" 
                        required 
                        min="10000" 
                        class="w-full px-4 py-3 bg-gray-700/50 border border-gray-600 text-white rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-300 pr-12">
                    <span class="absolute right-4 top-1/2 transform -translate-y-1/2 text-gray-400">원</span>
                </div>
            </div>
            
            <!-- 계산 정보 -->
            <div class="bg-gray-700/30 rounded-xl p-4 space-y-2">
                <div class="flex justify-between items-center text-sm">
                    <span class="text-gray-400">요청 금액</span>
                    <span class="text-white font-medium" id="requestAmount">0원</span>
                </div>
                <div class="flex justify-between items-center text-sm">
                    <span class="text-gray-400">수수료</span>
                    <span class="text-red-400 font-medium">-1,000원</span>
                </div>
                <div class="border-t border-gray-600 pt-2">
                    <div class="flex justify-between items-center">
                        <span class="text-gray-300 font-medium">실제 입금액</span>
                        <span class="text-green-400 font-bold" id="actualAmount">0원</span>
                    </div>
                </div>
            </div>
            
            <button type="submit" 
                id="submitBtn"
                class="w-full bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-semibold py-3 px-6 rounded-xl transition-all duration-300 transform hover:scale-[1.02] disabled:opacity-50 disabled:cursor-not-allowed disabled:transform-none shadow-lg hover:shadow-xl">
                <span class="flex items-center justify-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"></path>
                    </svg>
                    출금 신청하기
                </span>
            </button>
        </form>
    </div>
</div>

<main class="flex-1 p-6 space-y-8 bg-gradient-to-br from-gray-900 via-gray-800 to-gray-900 min-h-screen">
    <!-- 대시보드 요약 -->
    <div class="grid grid-cols-1 md:grid-cols-3 gap-6 max-w-6xl mx-auto">
        <!-- 출금 가능 금액 카드 -->
        <div class="bg-gradient-to-br from-green-500/10 to-emerald-600/10 border border-green-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-green-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z"></path>
                    </svg>
                </div>
                <button onclick="openWithdrawalModal()" 
                    class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-all duration-300 transform hover:scale-105">
                    출금 신청
                </button>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">출금 가능 금액</h3>
                <p class="text-2xl font-bold text-white" id="dashboardAmount">0원</p>
            </div>
        </div>

        <!-- 오늘 출금 신청 카드 -->
        <div class="bg-gradient-to-br from-blue-500/10 to-blue-600/10 border border-blue-500/20 rounded-xl p-6">
            <div class="flex items-center mb-4">
                <div class="w-12 h-12 bg-blue-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">오늘 출금 신청</h3>
                <p class="text-2xl font-bold text-white" id="todayCount">0건</p>
            </div>
        </div>

        <!-- 대기중인 출금 카드 -->
        <div class="bg-gradient-to-br from-orange-500/10 to-orange-600/10 border border-orange-500/20 rounded-xl p-6">
            <div class="flex items-center mb-4">
                <div class="w-12 h-12 bg-orange-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-orange-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">처리 대기중</h3>
                <p class="text-2xl font-bold text-white" id="pendingAmount">0원</p>
            </div>
        </div>
    </div>

    <!-- 출금 내역 -->
    <div class="w-full">
        <div class="bg-gradient-to-br from-gray-800 to-gray-900 rounded-2xl shadow-2xl border border-gray-700/50 overflow-hidden">
            <div class="p-6 border-b border-gray-700/50">
                <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                    <div>
                        <h3 class="text-2xl font-bold text-white">출금 신청 내역</h3>
                        <p class="text-gray-400 text-sm mt-1">출금 신청 현황을 확인하세요</p>
                    </div>
                    <button onclick="loadWithdrawals()" class="flex items-center px-4 py-2 text-sm text-blue-400 hover:text-blue-300 hover:bg-blue-500/10 rounded-lg transition-all duration-300">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                        </svg>
                        새로고침
                    </button>
                </div>
                
                <!-- 검색 필터 -->
                <div class="mt-6 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">상태</label>
                        <select id="statusFilter" onchange="loadWithdrawals()" 
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                            <option value="">전체 상태</option>
                            <option value="requested">신청</option>
                            <option value="pending">대기</option>
                            <option value="completed">완료</option>
                            <option value="cancelled">취소</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">시작일</label>
                        <input type="date" id="startDate" onchange="loadWithdrawals()" 
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">종료일</label>
                        <input type="date" id="endDate" onchange="loadWithdrawals()" 
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                    </div>
                    
                    <div class="flex items-end">
                        <button onclick="resetDateFilter()" 
                            class="w-full px-4 py-2 bg-gray-600 hover:bg-gray-500 text-white rounded-lg transition-all duration-300 text-sm">
                            초기화
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-gray-700/50">
                        <tr>
                            <th class="px-6 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider">신청일시</th>
                            <th class="px-6 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider">금액 정보</th>
                            <th class="px-6 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider">상태</th>
                            <th class="px-6 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider">처리일시</th>
                            <th class="px-6 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider">관리</th>
                        </tr>
                    </thead>
                    <tbody id="withdrawalTableBody" class="divide-y divide-gray-700/30">
                        <!-- 데이터는 AJAX로 로드 -->
                    </tbody>
                </table>
            </div>
            
            <!-- 로딩 표시 -->
            <div id="loadingIndicator" class="hidden py-12 text-center">
                <div class="inline-flex items-center px-4 py-2 font-semibold leading-6 text-sm text-blue-400">
                    <svg class="animate-spin -ml-1 mr-3 h-5 w-5 text-blue-400" fill="none" viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                    </svg>
                    데이터를 불러오는 중...
                </div>
            </div>
            
            <!-- 페이지네이션 -->
            <div id="pagination" class="p-6 border-t border-gray-700/30">
                <!-- 페이지네이션은 JavaScript로 동적 생성 -->
            </div>
        </div>
    </div>
</main>

<script>
// 전역 변수
// 전역 변수
let currentPage = 1;
let availableAmount = 0;
let withdrawalRestriction = null; // 출금 제한 정보
const FEE = 1000; // 고정 수수료

// 페이지 로드 시 초기화
document.addEventListener('DOMContentLoaded', function() {
    // 오늘 날짜를 기본값으로 설정
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('startDate').value = today;
    document.getElementById('endDate').value = today;
    
    loadPartnerInfo();
    loadWithdrawalRestriction(); // 출금 제한 정보 로드
    loadWithdrawals();
    
    // 금액 입력 시 실제 입금액 계산
    document.getElementById('withdrawAmount').addEventListener('input', calculateActualAmount);
    
    // 출금 신청 폼 제출
    document.getElementById('withdrawalForm').addEventListener('submit', submitWithdrawal);
    
    // 모달 외부 클릭 시 닫기
    document.getElementById('withdrawalModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeWithdrawalModal();
        }
    });
});

// 출금 제한 정보 로드
function loadWithdrawalRestriction() {
    fetch('/ajax/get_withdrawal_restriction.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                withdrawalRestriction = data;
                updateWithdrawalUI();
            } else {
                console.error('출금 제한 정보 로드 실패:', data.message);
            }
        })
        .catch(error => {
            console.error('출금 제한 정보 로드 오류:', error);
        });
}

// 출금 UI 업데이트 (제한 시간 안내)
function updateWithdrawalUI() {
    if (!withdrawalRestriction) return;
    
    const { is_blocked, block_info, current_time } = withdrawalRestriction;
    
    // 기존 안내 메시지 제거
    const existingNotice = document.getElementById('withdrawal-time-notice');
    if (existingNotice) {
        existingNotice.remove();
    }
    
    // 출금 차단 시간 안내 추가
    if (block_info) {
        const modalContent = document.querySelector('#withdrawalModal .bg-gradient-to-br');
        const bankInfoDiv = document.getElementById('bankInfo');
        
        const timeNotice = document.createElement('div');
        timeNotice.id = 'withdrawal-time-notice';
        timeNotice.className = `${is_blocked ? 'bg-red-500/10 border-red-500/20' : 'bg-blue-500/10 border-blue-500/20'} rounded-xl p-4 mb-6 border`;
        
        const statusIcon = is_blocked 
            ? '<svg class="w-5 h-5 text-red-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>'
            : '<svg class="w-5 h-5 text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
        
        const { start_time, end_time, grade_name } = block_info;
        const timeRangeText = getTimeRangeDescription(start_time, end_time);
        
        timeNotice.innerHTML = `
            <div class="flex items-start">
                <div class="flex-shrink-0 mt-1">
                    ${statusIcon}
                </div>
                <div class="ml-3 flex-1">
                    <h4 class="text-sm font-semibold ${is_blocked ? 'text-red-300' : 'text-blue-300'} mb-2">
                        ${is_blocked ? '⛔ 출금 차단 시간' : '📅 출금 차단 시간 안내'}
                    </h4>
                    <div class="text-xs space-y-1">
                        <div class="flex justify-between items-center">
                            <span class="${is_blocked ? 'text-red-200' : 'text-blue-200'}">등급</span>
                            <span class="${is_blocked ? 'text-red-100' : 'text-blue-100'} font-medium">${grade_name}</span>
                        </div>
                        <div class="flex justify-between items-center">
                            <span class="${is_blocked ? 'text-red-200' : 'text-blue-200'}">차단 시간</span>
                            <span class="${is_blocked ? 'text-red-100' : 'text-blue-100'} font-medium">${timeRangeText}</span>
                        </div>
                        <div class="flex justify-between items-center">
                            <span class="${is_blocked ? 'text-red-200' : 'text-blue-200'}">현재 시간</span>
                            <span class="${is_blocked ? 'text-red-100' : 'text-blue-100'} font-medium">${current_time}</span>
                        </div>
                        ${is_blocked ? `
                            <div class="mt-2 pt-2 border-t border-red-500/20">
                                <p class="text-red-200 font-medium">현재 출금 신청이 제한된 시간입니다.</p>
                                <p class="text-red-300 text-xs">차단 해제 시간: ${end_time}</p>
                            </div>
                        ` : `
                            <div class="mt-2 pt-2 border-t border-blue-500/20">
                                <p class="text-blue-200">위 시간대에는 출금 신청이 제한됩니다.</p>
                            </div>
                        `}
                    </div>
                </div>
            </div>
        `;
        
        // bankInfo 위에 삽입
        modalContent.insertBefore(timeNotice, bankInfoDiv);
    }
    
    // 출금 신청 버튼 상태 업데이트
    updateSubmitButtonState();
}

// 출금 신청 버튼 상태 업데이트
function updateSubmitButtonState() {
    const submitBtn = document.getElementById('submitBtn');
    const withdrawInput = document.getElementById('withdrawAmount');
    
    if (withdrawalRestriction && withdrawalRestriction.is_blocked) {
        submitBtn.disabled = true;
        submitBtn.innerHTML = `
            <span class="flex items-center justify-center">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                </svg>
                출금 제한 시간
            </span>
        `;
        withdrawInput.disabled = true;
    } else {
        submitBtn.disabled = false;
        submitBtn.innerHTML = `
            <span class="flex items-center justify-center">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"></path>
                </svg>
                출금 신청하기
            </span>
        `;
        withdrawInput.disabled = false;
    }
}

// 시간 범위 설명 생성
function getTimeRangeDescription(startTime, endTime) {
    if (!startTime || !endTime) return '';
    
    const [startHour] = startTime.split(':').map(Number);
    const [endHour] = endTime.split(':').map(Number);
    
    if (endHour <= startHour) {
        return `${startTime} ~ ${endTime} (다음날)`;
    }
    return `${startTime} ~ ${endTime}`;
}

// 출금 신청 모달 열기
function openWithdrawalModal() {
    const modal = document.getElementById('withdrawalModal');
    modal.classList.remove('hidden');
    modal.classList.add('flex');
    document.body.style.overflow = 'hidden';
    
    // 모달이 열릴 때마다 최신 정보 로드
    loadPartnerInfo();
    loadWithdrawalRestriction(); // 출금 제한 정보도 새로고침
}

// 출금 신청 제출 (제한 시간 체크 추가)
function submitWithdrawal(e) {
    e.preventDefault();
    
    // 출금 제한 시간 체크
    if (withdrawalRestriction && withdrawalRestriction.is_blocked) {
        const { block_info } = withdrawalRestriction;
        const timeRange = getTimeRangeDescription(block_info.start_time, block_info.end_time);
        showAlert('error', `현재 출금 제한 시간입니다.\n제한 시간: ${timeRange}\n차단 해제 시간: ${block_info.end_time}`);
        return;
    }
    
    const amount = parseInt(document.getElementById('withdrawAmount').value);
    const submitBtn = document.getElementById('submitBtn');
    
    // 기존 유효성 검사
    if (!amount || amount <= 0) {
        showAlert('error', '올바른 금액을 입력해주세요.');
        return;
    }
    
    if (amount < 10000) {
        showAlert('error', '최소 출금 금액은 10,000원입니다.');
        return;
    }
    
    if (availableAmount <= 0) {
        showAlert('error', '출금 가능한 잔액이 없습니다.');
        return;
    }
    
    if (amount > availableAmount) {
        showAlert('error', '출금 가능 금액을 초과했습니다.');
        return;
    }
    
    // 확인 대화상자
    const actualAmount = amount - FEE;
    if (!confirm(`출금 신청 확인\n\n요청 금액: ${amount.toLocaleString()}원\n수수료: ${FEE.toLocaleString()}원\n실제 입금액: ${actualAmount.toLocaleString()}원\n\n정말로 출금 신청하시겠습니까?`)) {
        return;
    }
    
    // 서버 측에서도 한번 더 체크하므로 이후 로직은 기존과 동일
    // 버튼 비활성화
    submitBtn.disabled = true;
    submitBtn.innerHTML = `
        <span class="flex items-center justify-center">
            <svg class="animate-spin -ml-1 mr-3 h-4 w-4 text-white" fill="none" viewBox="0 0 24 24">
                <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l-3-2.647z"></path>
            </svg>
            처리 중...
        </span>
    `;
    
    // AJAX 요청
    fetch('/ajax/process_partner_withdrawal.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            amount: amount
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('success', '출금 신청이 완료되었습니다!');
            closeWithdrawalModal();
            loadPartnerInfo(); // 잔액 새로고침
            loadWithdrawals(); // 내역 새로고침
        } else {
            showAlert('error', data.message || '출금 신청에 실패했습니다.');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('error', '서버 오류가 발생했습니다.');
    })
    .finally(() => {
        updateSubmitButtonState(); // 버튼 상태 복원
    });
}

// 출금 신청 모달 닫기
function closeWithdrawalModal() {
    const modal = document.getElementById('withdrawalModal');
    modal.classList.add('hidden');
    modal.classList.remove('flex');
    document.body.style.overflow = '';
    
    // 폼 초기화
    document.getElementById('withdrawalForm').reset();
    calculateActualAmount();
}

// 파트너 정보 로드 (잔액, 계좌정보)
function loadPartnerInfo() {
    fetch('/ajax/fetch_partner_info.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // 잔액 표시 (대시보드와 모달 양쪽)
                availableAmount = data.amount;
                document.getElementById('availableAmount').textContent = Number(availableAmount).toLocaleString() + '원';
                document.getElementById('dashboardAmount').textContent = Number(availableAmount).toLocaleString() + '원';
                
                // max 속성 설정 (0 이하인 경우 처리)
                const withdrawInput = document.getElementById('withdrawAmount');
                if (availableAmount > 0) {
                    withdrawInput.max = availableAmount;
                    withdrawInput.disabled = false;
                } else {
                    withdrawInput.disabled = true;
                    // showAlert('error', '출금 가능한 잔액이 없습니다.');
                }
                
                // 계좌 정보 표시
                if (data.bankInfo) {
                    document.getElementById('bankInfo').innerHTML = `
                        <div class="flex items-start">
                            <div class="flex-shrink-0">
                                <div class="w-8 h-8 bg-green-500/20 rounded-lg flex items-center justify-center">
                                    <svg class="w-4 h-4 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"></path>
                                    </svg>
                                </div>
                            </div>
                            <div class="ml-3 flex-1">
                                <h4 class="text-sm font-medium text-white mb-2">등록된 계좌</h4>
                                <div class="space-y-1 text-xs">
                                    <div class="flex justify-between">
                                        <span class="text-gray-400">은행</span>
                                        <span class="text-white">${escapeHtml(data.bankInfo.bank)}</span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span class="text-gray-400">계좌번호</span>
                                        <span class="text-white">${escapeHtml(data.bankInfo.account_number)}</span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span class="text-gray-400">예금주</span>
                                        <span class="text-white">${escapeHtml(data.bankInfo.account_holder)}</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    `;
                } else {
                    document.getElementById('bankInfo').innerHTML = `
                        <div class="text-center py-4">
                            <div class="w-8 h-8 bg-yellow-500/20 rounded-lg flex items-center justify-center mx-auto mb-2">
                                <svg class="w-4 h-4 text-yellow-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.664-.833-2.464 0L3.34 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                                </svg>
                            </div>
                            <p class="text-yellow-400 text-sm font-medium mb-2">계좌 정보 미등록</p>
                            <a href="/partner/settings.php" class="inline-flex items-center px-3 py-1 bg-blue-600 hover:bg-blue-700 text-white text-xs rounded-lg transition-all duration-300">
                                등록하기
                            </a>
                        </div>
                    `;
                }
            } else {
                showAlert('error', data.message || '정보를 불러올 수 없습니다.');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showAlert('error', '서버 오류가 발생했습니다.');
        });
}

// 출금 내역 로드
function loadWithdrawals(page = 1) {
    currentPage = page;
    const status = document.getElementById('statusFilter').value;
    const startDate = document.getElementById('startDate').value;
    const endDate = document.getElementById('endDate').value;
    
    const tbody = document.getElementById('withdrawalTableBody');
    const loadingIndicator = document.getElementById('loadingIndicator');
    
    // 로딩 표시
    tbody.style.display = 'none';
    loadingIndicator.classList.remove('hidden');
    
    const params = new URLSearchParams({
        page: currentPage,
        status: status,
        start_date: startDate,
        end_date: endDate
    });
    
    fetch(`/ajax/fetch_partner_withdrawals.php?${params.toString()}`)
        .then(response => response.json())
        .then(data => {
            tbody.style.display = '';
            loadingIndicator.classList.add('hidden');
            
            if (data.success) {
                renderWithdrawals(data.withdrawals);
                renderPagination(data.totalPages, data.currentPage);
                
                // 대시보드 통계 업데이트
                if (data.stats) {
                    const todayWithdrawals = data.withdrawals.filter(w => {
                        const today = new Date().toDateString();
                        const createdDate = new Date(w.created_at).toDateString();
                        return today === createdDate;
                    });
                    
                    document.getElementById('todayCount').textContent = todayWithdrawals.length + '건';
                    document.getElementById('pendingAmount').textContent = 
                        Number(data.stats.total_pending_amount || 0).toLocaleString() + '원';
                }
            } else {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="5" class="px-6 py-12 text-center">
                            <div class="text-gray-400">
                                <svg class="w-12 h-12 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 13V6a2 2 0 00-2-2H6a2 2 0 00-2 2v7m16 0v5a2 2 0 01-2 2H6a2 2 0 01-2-2v-5m16 0h-2.586a1 1 0 00-.707.293l-2.414 2.414a1 1 0 01-.707.293h-3.172a1 1 0 01-.707-.293l-2.414-2.414A1 1 0 006.586 13H4"></path>
                                </svg>
                                <p class="text-lg font-medium">${escapeHtml(data.message || '데이터를 불러올 수 없습니다.')}</p>
                            </div>
                        </td>
                    </tr>
                `;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            tbody.style.display = '';
            loadingIndicator.classList.add('hidden');
            tbody.innerHTML = `
                <tr>
                    <td colspan="5" class="px-6 py-12 text-center text-red-400">
                        <svg class="w-12 h-12 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                        <p class="text-lg font-medium">서버 오류가 발생했습니다.</p>
                    </td>
                </tr>
            `;
        });
}

// 출금 내역 렌더링
function renderWithdrawals(withdrawals) {
    const tbody = document.getElementById('withdrawalTableBody');
    
    if (!withdrawals || withdrawals.length === 0) {
        tbody.innerHTML = `
            <tr>
                <td colspan="5" class="px-6 py-12 text-center">
                    <div class="text-gray-400">
                        <svg class="w-16 h-16 mx-auto mb-4 opacity-30" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                        </svg>
                        <p class="text-xl font-medium mb-2">출금 내역이 없습니다</p>
                        <p class="text-sm opacity-75">선택한 기간에 해당하는 출금 신청 내역이 없습니다.</p>
                    </div>
                </td>
            </tr>
        `;
        return;
    }
    
    tbody.innerHTML = withdrawals.map(withdrawal => {
        const statusInfo = getStatusInfo(withdrawal.status);
        
        return `
            <tr class="hover:bg-gray-700/20 transition-all duration-200">
                <td class="px-6 py-4">
                    <div class="text-sm">
                        <div class="text-white font-medium">${formatDateTime(withdrawal.created_at).date}</div>
                        <div class="text-gray-400 text-xs">${formatDateTime(withdrawal.created_at).time}</div>
                    </div>
                </td>
                <td class="px-6 py-4">
                    <div class="text-sm">
                        <div class="text-white font-semibold">${Number(withdrawal.amount).toLocaleString()}원</div>
                        <div class="text-gray-400 text-xs">실제 출금: ${(Number(withdrawal.amount) - 1000).toLocaleString()}원</div>
                    </div>
                </td>
                <td class="px-6 py-4">
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${statusInfo.class}">
                        ${statusInfo.icon}
                        ${statusInfo.text}
                    </span>
                </td>
                <td class="px-6 py-4 text-sm">
                    ${withdrawal.completed_at ? `
                        <div class="text-white">${formatDateTime(withdrawal.completed_at).date}</div>
                        <div class="text-gray-400 text-xs">${formatDateTime(withdrawal.completed_at).time}</div>
                    ` : '<span class="text-gray-500">-</span>'}
                </td>
                <td class="px-6 py-4">
                    ${withdrawal.status === 'requested' ? `
                        <button onclick="cancelWithdrawal(${withdrawal.id})" 
                            class="inline-flex items-center px-3 py-1 text-xs text-red-400 hover:text-red-300 hover:bg-red-500/10 rounded-lg transition-all duration-300">
                            <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                            취소
                        </button>
                    ` : '<span class="text-gray-500">-</span>'}
                </td>
            </tr>
        `;
    }).join('');
}

// 페이지네이션 렌더링
function renderPagination(totalPages, currentPage) {
    const pagination = document.getElementById('pagination');
    
    if (totalPages <= 1) {
        pagination.innerHTML = '';
        return;
    }
    
    let html = '<div class="flex items-center justify-center space-x-2">';
    
    // 이전 버튼
    if (currentPage > 1) {
        html += `
            <button onclick="loadWithdrawals(${currentPage - 1})" 
                class="flex items-center px-3 py-2 text-sm text-gray-400 hover:text-white hover:bg-gray-700 rounded-lg transition-all duration-300">
                <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                </svg>
                이전
            </button>
        `;
    }
    
    // 페이지 번호
    for (let i = Math.max(1, currentPage - 2); i <= Math.min(totalPages, currentPage + 2); i++) {
        html += `
            <button onclick="loadWithdrawals(${i})" 
                class="px-3 py-2 text-sm rounded-lg transition-all duration-300 ${
                    i === currentPage 
                        ? 'bg-gradient-to-r from-blue-600 to-purple-600 text-white font-medium' 
                        : 'text-gray-400 hover:text-white hover:bg-gray-700'
                }">
                ${i}
            </button>
        `;
    }
    
    // 다음 버튼
    if (currentPage < totalPages) {
        html += `
            <button onclick="loadWithdrawals(${currentPage + 1})" 
                class="flex items-center px-3 py-2 text-sm text-gray-400 hover:text-white hover:bg-gray-700 rounded-lg transition-all duration-300">
                다음
                <svg class="w-4 h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </button>
        `;
    }
    
    html += '</div>';
    pagination.innerHTML = html;
}

// 실제 입금액 계산
function calculateActualAmount() {
    const amount = parseInt(document.getElementById('withdrawAmount').value) || 0;
    const actualAmount = Math.max(0, amount - FEE);
    
    document.getElementById('requestAmount').textContent = amount.toLocaleString() + '원';
    document.getElementById('actualAmount').textContent = actualAmount.toLocaleString() + '원';
}
// 나머지 기존 함수들은 그대로 유지...
// (loadPartnerInfo, loadWithdrawals, calculateActualAmount 등)

// 출금 취소
function cancelWithdrawal(id) {
    if (!confirm('출금 신청을 취소하시겠습니까?\n\n취소 후에는 되돌릴 수 없습니다.')) {
        return;
    }
    
    fetch('/ajax/cancel_partner_withdrawal.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            id: id
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('success', '출금 신청이 취소되었습니다.');
            loadPartnerInfo(); // 잔액 새로고침
            loadWithdrawals(); // 내역 새로고침
        } else {
            showAlert('error', data.message || '취소 처리에 실패했습니다.');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('error', '서버 오류가 발생했습니다.');
    });
}

// 날짜 필터 초기화
function resetDateFilter() {
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('startDate').value = today;
    document.getElementById('endDate').value = today;
    loadWithdrawals();
}

// 상태 정보
function getStatusInfo(status) {
    const statusMap = {
        'requested': {
            class: 'bg-yellow-500/20 text-yellow-400 border border-yellow-500/30',
            text: '신청',
            icon: '<svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>'
        },
        'pending': {
            class: 'bg-orange-500/20 text-orange-400 border border-orange-500/30',
            text: '처리중',
            icon: '<svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path></svg>'
        },
        'completed': {
            class: 'bg-green-500/20 text-green-400 border border-green-500/30',
            text: '승인',
            icon: '<svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>'
        },
        'cancelled': {
            class: 'bg-red-500/20 text-red-400 border border-red-500/30',
            text: '취소',
            icon: '<svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>'
        }
    };
    
    return statusMap[status] || {
        class: 'bg-gray-500/20 text-gray-400 border border-gray-500/30',
        text: status,
        icon: ''
    };
}

// 날짜시간 포맷팅
function formatDateTime(dateTimeString) {
    const date = new Date(dateTimeString);
    const dateStr = date.toLocaleDateString('ko-KR', { 
        year: 'numeric', 
        month: '2-digit', 
        day: '2-digit' 
    });
    const timeStr = date.toLocaleTimeString('ko-KR', { 
        hour: '2-digit', 
        minute: '2-digit' 
    });
    
    return { date: dateStr, time: timeStr };
}

// HTML 이스케이프
function escapeHtml(text) {
    if (!text) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}

// 알림 표시
function showAlert(type, message) {
    // 기존 알림 제거
    const existingAlert = document.querySelector('.alert-message');
    if (existingAlert) {
        existingAlert.remove();
    }
    
    const alertClass = type === 'success' 
        ? 'bg-gradient-to-r from-green-600 to-emerald-600 border-green-500' 
        : 'bg-gradient-to-r from-red-600 to-rose-600 border-red-500';
    
    const alertIcon = type === 'success'
        ? '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>'
        : '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
    
    const alert = document.createElement('div');
    alert.className = `alert-message fixed top-4 right-4 ${alertClass} text-white px-6 py-4 rounded-lg shadow-2xl z-[60] border flex items-center animate-slide-in`;
    alert.innerHTML = `
        <div class="flex items-center">
            ${alertIcon}
            <span class="ml-3 font-medium">${message}</span>
        </div>
    `;
    
    document.body.appendChild(alert);
    
    setTimeout(() => {
        alert.classList.add('animate-fade-out');
        setTimeout(() => alert.remove(), 300);
    }, 3000);
}

// 모바일 사이드바 토글
function toggleSidebar() {
    const sidebar = document.querySelector('.sidebar');
    if (sidebar) {
        sidebar.classList.toggle('hidden');
    }
}
</script>

<style>
@keyframes slide-in {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes fade-out {
    from {
        opacity: 1;
    }
    to {
        opacity: 0;
    }
}

.animate-slide-in {
    animation: slide-in 0.3s ease-out;
}

.animate-fade-out {
    animation: fade-out 0.3s ease-out;
}

/* 커스텀 스크롤바 */
.overflow-x-auto::-webkit-scrollbar {
    height: 8px;
}

.overflow-x-auto::-webkit-scrollbar-track {
    background: rgba(55, 65, 81, 0.3);
    border-radius: 4px;
}

.overflow-x-auto::-webkit-scrollbar-thumb {
    background: rgba(107, 114, 128, 0.5);
    border-radius: 4px;
}

.overflow-x-auto::-webkit-scrollbar-thumb:hover {
    background: rgba(107, 114, 128, 0.7);
}

/* 모달 스크롤바 */
#withdrawalModal .max-h-\[90vh\]::-webkit-scrollbar {
    width: 6px;
}

#withdrawalModal .max-h-\[90vh\]::-webkit-scrollbar-track {
    background: rgba(55, 65, 81, 0.3);
    border-radius: 3px;
}

#withdrawalModal .max-h-\[90vh\]::-webkit-scrollbar-thumb {
    background: rgba(107, 114, 128, 0.5);
    border-radius: 3px;
}
</style>

</body>
</html>