<?php
include 'layout/header.php';
?>

<!-- 모바일 메뉴 열기 버튼 -->
<button class="md:hidden fixed top-4 left-4 z-40 bg-gradient-to-r from-blue-600 to-purple-600 text-white p-3 rounded-full shadow-lg hover:shadow-xl transition-all duration-300" onclick="toggleSidebar()">
    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
    </svg>
</button>

<main class="flex-1 p-6 space-y-8 bg-gradient-to-br from-gray-900 via-gray-800 to-gray-900 min-h-screen">
    <!-- 대시보드 요약 -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <!-- 오늘 출금 신청 -->
        <div class="bg-gradient-to-br from-blue-500/10 to-blue-600/10 border border-blue-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-blue-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">오늘 출금 신청</h3>
                <p class="text-2xl font-bold text-white" id="todayCount">0건</p>
                <p class="text-sm text-blue-400" id="todayAmount">0원</p>
            </div>
        </div>

        <!-- 처리 대기중 -->
        <div class="bg-gradient-to-br from-yellow-500/10 to-yellow-600/10 border border-yellow-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-yellow-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-yellow-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">처리 대기중</h3>
                <p class="text-2xl font-bold text-white" id="pendingCount">0건</p>
                <p class="text-sm text-yellow-400" id="pendingAmount">0원</p>
            </div>
        </div>

        <!-- 오늘 처리 완료 -->
        <div class="bg-gradient-to-br from-green-500/10 to-green-600/10 border border-green-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-green-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">오늘 처리 완료</h3>
                <p class="text-2xl font-bold text-white" id="completedCount">0건</p>
                <p class="text-sm text-green-400" id="completedAmount">0원</p>
            </div>
        </div>

        <!-- 총 수수료 -->
        <div class="bg-gradient-to-br from-purple-500/10 to-purple-600/10 border border-purple-500/20 rounded-xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div class="w-12 h-12 bg-purple-500/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2-2V7a2 2 0 012-2h2a2 2 0 002 2v2a2 2 0 002 2h2a2 2 0 012-2V7a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 00-2 2h-2a2 2 0 00-2 2v6"></path>
                    </svg>
                </div>
            </div>
            <div>
                <h3 class="text-sm font-medium text-gray-400 mb-1">오늘 총 수수료</h3>
                <p class="text-2xl font-bold text-white" id="totalFees">0원</p>
            </div>
        </div>
    </div>

    <!-- 출금 신청 내역 -->
    <div class="w-full">
        <div class="bg-gradient-to-br from-gray-800 to-gray-900 rounded-2xl shadow-2xl border border-gray-700/50 overflow-hidden">
            <div class="p-6 border-b border-gray-700/50">
                <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-6">
                    <div>
                        <h3 class="text-2xl font-bold text-white">출금 신청 내역</h3>
                        <p class="text-gray-400 text-sm mt-1">출금 신청 현황을 관리하세요</p>
                    </div>
                    <div class="flex items-center space-x-3">
                        <!-- 엑셀 다운로드 버튼 추가 -->
                        <button onclick="downloadExcel()" class="flex items-center px-4 py-2 text-sm text-green-400 hover:text-green-300 hover:bg-green-500/10 rounded-lg transition-all duration-300">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                            </svg>
                            엑셀 다운로드
                        </button>
                        
                        <button onclick="fetchWithdrawals()" class="flex items-center px-4 py-2 text-sm text-blue-400 hover:text-blue-300 hover:bg-blue-500/10 rounded-lg transition-all duration-300">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                            </svg>
                            새로고침
                        </button>
                    </div>
                </div>
                
                <!-- 검색 필터 -->
                <form id="searchForm" class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-6 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">시작일</label>
                        <input type="date" name="start_date" id="start_date"
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">종료일</label>
                        <input type="date" name="end_date" id="end_date"
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">회원명/ID</label>
                        <input type="text" name="search" id="search" placeholder="회원명 또는 ID"
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">가맹점</label>
                        <div class="relative">
                            <input type="hidden" name="store_code" id="store_code" value="">
                            <input type="text" 
                                id="store_search" 
                                placeholder="가맹점 검색..."
                                autocomplete="off"
                                class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                            <div id="store_dropdown" 
                                class="absolute z-10 w-full mt-1 bg-gray-800 border border-gray-600 rounded-md shadow-lg hidden max-h-60 overflow-y-auto">
                                <!-- 동적으로 로드됨 -->
                            </div>
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-400 mb-1">상태</label>
                        <select name="status" id="status"
                            class="w-full px-3 py-2 bg-gray-700/50 border border-gray-600 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-300">
                            <option value="">전체 상태</option>
                            <option value="requested">신청</option>
                            <option value="pending">대기</option>
                            <option value="completed">완료</option>
                            <option value="cancelled">취소</option>
                        </select>
                    </div>
                    
                    <div class="flex items-end space-x-2">
                        <button type="submit" 
                            class="flex-1 px-4 py-2 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white rounded-lg transition-all duration-300">
                            검색
                        </button>
                        <button type="button" onclick="resetForm()"
                            class="px-4 py-2 bg-gray-600 hover:bg-gray-500 text-white rounded-lg transition-all duration-300">
                            초기화
                        </button>
                    </div>
                </form>
            </div>
            
            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-gray-700/50">
                        <tr>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('id')">
                                ID
                                <span class="sort-indicator ml-1" data-column="id">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('user_name')">
                                회원명
                                <span class="sort-indicator ml-1" data-column="user_name">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('member_user_id')">
                                회원 ID
                                <span class="sort-indicator ml-1" data-column="member_user_id">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-right text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('amount')">
                                신청금액
                                <span class="sort-indicator ml-1" data-column="amount">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-right text-xs font-medium text-gray-400 uppercase tracking-wider">수수료</th>
                            <th class="px-4 py-4 text-right text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('actual_amount')">
                                실제 출금액
                                <span class="sort-indicator ml-1" data-column="actual_amount">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('store_code')">
                                가맹점
                                <span class="sort-indicator ml-1" data-column="store_code">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider">계좌정보</th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider">잔액</th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('status')">
                                상태
                                <span class="sort-indicator ml-1" data-column="status">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('created_at')">
                                신청일시
                                <span class="sort-indicator ml-1" data-column="created_at">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-left text-xs font-medium text-gray-400 uppercase tracking-wider cursor-pointer hover:bg-gray-600/50 transition-all duration-200" 
                                onclick="sortTable('completed_at')">
                                처리일시
                                <span class="sort-indicator ml-1" data-column="completed_at">⇅</span>
                            </th>
                            <th class="px-4 py-4 text-center text-xs font-medium text-gray-400 uppercase tracking-wider">관리</th>
                        </tr>
                    </thead>
                    <tbody id="withdrawalTableBody" class="divide-y divide-gray-700/30">
                        <!-- 데이터는 JavaScript로 동적 로드 -->
                    </tbody>
                </table>
            </div>
            
            <!-- 로딩 표시 -->
            <div id="loadingIndicator" class="hidden py-12 text-center">
                <div class="inline-flex items-center px-4 py-2 font-semibold leading-6 text-sm text-blue-400">
                    <svg class="animate-spin -ml-1 mr-3 h-5 w-5 text-blue-400" fill="none" viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l-3-2.647z"></path>
                    </svg>
                    데이터를 불러오는 중...
                </div>
            </div>
            
            <!-- 페이지네이션 -->
            <div id="pagination" class="p-6 border-t border-gray-700/30">
                <!-- 페이지네이션은 JavaScript로 동적 생성 -->
            </div>
        </div>
    </div>
</main>

<script>
// 전역 변수
let currentPage = 1;
let isLoading = false;
let stores = []; // 가맹점 목록 저장
let currentSort = { column: 'created_at', direction: 'desc' }; // 정렬 상태
let withdrawalData = []; // 현재 데이터 저장
const WITHDRAWAL_FEE = 1000; // 고정 수수료

// 페이지 로드 시 초기화
document.addEventListener('DOMContentLoaded', function() {
    // 어제~오늘을 기본값으로 설정
    const today = new Date().toISOString().split('T')[0];
    const yesterday = new Date();
    yesterday.setDate(yesterday.getDate() - 1);
    const yesterdayStr = yesterday.toISOString().split('T')[0];
    
    document.getElementById('start_date').value = yesterdayStr;
    document.getElementById('end_date').value = today;
    
    loadStores(); // 가맹점 목록 로드
    setFormValues(); // 폼 값 설정
    fetchWithdrawals();
    
    // 검색 폼 제출 이벤트
    document.getElementById('searchForm').addEventListener('submit', function(e) {
        e.preventDefault();
        currentPage = 1;
        fetchWithdrawals();
    });
    
    // 자동 새로고침 (30초마다)
    setInterval(function() {
        if (!isLoading) {
            fetchWithdrawals();
        }
    }, 30000);
});

// 유틸리티 함수들
function escapeHtml(text) {
    if (text == null) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return String(text).replace(/[&<>"']/g, m => map[m]);
}

function formatDate(date) {
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
}

function getDefaultDates() {
    const today = new Date();
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);
    
    return {
        startDate: formatDate(yesterday),
        endDate: formatDate(today)
    };
}

// 가맹점 목록 로드
function loadStores() {
    fetch('/ajax/fetch_store_drop.php')
        .then(response => response.json())
        .then(data => {
            if (data.success && data.stores) {
                stores = data.stores;
                initStoreSearch();
            }
        })
        .catch(error => {
            console.error('가맹점 목록 로드 실패:', error);
            // 실패 시 기본 목록 사용
            stores = [];
            initStoreSearch();
        });
}

// 가맹점 검색 초기화
function initStoreSearch() {
    const searchInput = document.getElementById('store_search');
    const dropdown = document.getElementById('store_dropdown');
    const storeCodeInput = document.getElementById('store_code');
    
    // 검색 입력 이벤트
    searchInput.addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        const filteredStores = stores.filter(store => 
            store.name.toLowerCase().includes(searchTerm) || 
            store.code.toLowerCase().includes(searchTerm)
        );
        
        showStoreDropdown(filteredStores);
    });
    
    // 포커스 이벤트
    searchInput.addEventListener('focus', function() {
        showStoreDropdown(stores);
    });
    
    // 외부 클릭 시 드롭다운 닫기
    document.addEventListener('click', function(e) {
        if (!searchInput.contains(e.target) && !dropdown.contains(e.target)) {
            dropdown.classList.add('hidden');
        }
    });
    
    // 입력창 비우기 시 선택 해제
    searchInput.addEventListener('keyup', function(e) {
        if (this.value === '') {
            storeCodeInput.value = '';
        }
    });
}

// 가맹점 드롭다운 표시
function showStoreDropdown(filteredStores) {
    const dropdown = document.getElementById('store_dropdown');
    
    if (filteredStores.length === 0) {
        dropdown.innerHTML = '<div class="px-4 py-2 text-gray-400">검색 결과가 없습니다</div>';
        dropdown.classList.remove('hidden');
        return;
    }
    
    let html = '<div class="px-4 py-2 hover:bg-gray-700 cursor-pointer text-gray-300" onclick="selectStore(null)">전체 가맹점</div>';
    
    filteredStores.forEach(store => {
        html += `
            <div class="px-4 py-2 hover:bg-gray-700 cursor-pointer" onclick="selectStore(${store.id}, '${escapeHtml(store.name)}', '${escapeHtml(store.code)}')">
                <span class="text-white">${escapeHtml(store.name)}</span>
                <span class="text-gray-400 ml-2">(${escapeHtml(store.code)})</span>
            </div>
        `;
    });
    
    dropdown.innerHTML = html;
    dropdown.classList.remove('hidden');
}

// 가맹점 선택
function selectStore(id, name, code) {
    const searchInput = document.getElementById('store_search');
    const storeCodeInput = document.getElementById('store_code');
    const dropdown = document.getElementById('store_dropdown');
    
    if (id === null) {
        searchInput.value = '';
        storeCodeInput.value = '';
    } else {
        searchInput.value = `${name} (${code})`;
        storeCodeInput.value = id;
    }
    
    dropdown.classList.add('hidden');
}

// 테이블 정렬 함수
function sortTable(column) {
    // 정렬 방향 토글
    if (currentSort.column === column) {
        currentSort.direction = currentSort.direction === 'asc' ? 'desc' : 'asc';
    } else {
        currentSort.column = column;
        currentSort.direction = 'desc';
    }
    
    // 정렬 아이콘 업데이트
    updateSortIcons();
    
    // 서버에서 정렬된 데이터 가져오기
    fetchWithdrawals();
}

// 정렬 아이콘 업데이트
function updateSortIcons() {
    // 모든 정렬 아이콘 초기화
    document.querySelectorAll('.sort-indicator').forEach(icon => {
        icon.textContent = '⇅';
        icon.classList.remove('text-blue-400');
    });
    
    // 현재 정렬 컬럼의 아이콘 업데이트
    const currentIcon = document.querySelector(`.sort-indicator[data-column="${currentSort.column}"]`);
    if (currentIcon) {
        currentIcon.classList.add('text-blue-400');
        if (currentSort.direction === 'asc') {
            currentIcon.textContent = '↑';
        } else {
            currentIcon.textContent = '↓';
        }
    }
}

// 폼 초기화
function resetForm() {
    const defaults = getDefaultDates();
    
    document.getElementById('searchForm').reset();
    document.getElementById('start_date').value = defaults.startDate;
    document.getElementById('end_date').value = defaults.endDate;
    
    // 정렬 초기화
    currentSort = { column: 'created_at', direction: 'desc' };
    updateSortIcons();
    
    currentPage = 1;
    fetchWithdrawals();
}

// URL 파라미터 가져오기
function getUrlParams() {
    const params = new URLSearchParams(window.location.search);
    return {
        page: parseInt(params.get('page')) || 1,
        search: params.get('search') || '',
        start_date: params.get('start_date') || '',
        end_date: params.get('end_date') || '',
        store_code: params.get('store_code') || '',
        status: params.get('status') || '',
        sort: params.get('sort') || 'created_at',
        order: params.get('order') || 'desc'
    };
}

// 폼 값 설정
function setFormValues() {
    const params = getUrlParams();
    const defaults = getDefaultDates();
    
    // URL에 날짜가 없으면 기본값 사용
    document.getElementById('search').value = params.search;
    document.getElementById('start_date').value = params.start_date || defaults.startDate;
    document.getElementById('end_date').value = params.end_date || defaults.endDate;
    document.getElementById('status').value = params.status;
    
    // 정렬 상태 설정
    currentSort.column = params.sort;
    currentSort.direction = params.order;
    updateSortIcons();
    
    currentPage = params.page;
}

// 출금 데이터 가져오기
function fetchWithdrawals() {
    if (isLoading) return;
    
    isLoading = true;
    const tbody = document.getElementById('withdrawalTableBody');
    const loadingIndicator = document.getElementById('loadingIndicator');
    
    // 로딩 표시
    tbody.style.display = 'none';
    loadingIndicator.classList.remove('hidden');
    
    // 폼 데이터 수집
    const formData = new FormData(document.getElementById('searchForm'));
    const params = new URLSearchParams();
    
    // 기본 날짜 설정
    const startDate = formData.get('start_date');
    const endDate = formData.get('end_date');
    const defaults = getDefaultDates();
    
    if (!startDate && !endDate) {
        params.append('start_date', defaults.startDate);
        params.append('end_date', defaults.endDate);
    } else {
        if (startDate) params.append('start_date', startDate);
        if (endDate) params.append('end_date', endDate);
    }
    
    // 나머지 파라미터 추가
    for (let [key, value] of formData.entries()) {
        if (value && key !== 'start_date' && key !== 'end_date') {
            params.append(key, value);
        }
    }
    
    // 정렬 파라미터 추가
    params.append('sort', currentSort.column);
    params.append('order', currentSort.direction);
    
    params.append('page', currentPage);
    params.append('per_page', 20);
    
    // API 호출
    fetch(`/ajax/fetch_withdrawals.php?${params.toString()}`)
        .then(response => response.json())
        .then(data => {
            tbody.style.display = '';
            loadingIndicator.classList.add('hidden');
            
            if (data.success) {
                renderWithdrawals(data.withdrawals || []);
                renderPagination(data.totalPages, data.currentPage);
                updateDashboard(data.withdrawals || []);
            } else {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="12" class="px-4 py-12 text-center">
                            <div class="text-gray-400">
                                <svg class="w-16 h-16 mx-auto mb-4 opacity-30" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 13V6a2 2 0 00-2-2H6a2 2 0 00-2 2v7m16 0v5a2 2 0 01-2 2H6a2 2 0 01-2-2v-5m16 0h-2.586a1 1 0 00-.707.293l-2.414 2.414a1 1 0 01-.707.293h-3.172a1 1 0 01-.707-.293l-2.414-2.414A1 1 0 006.586 13H4"></path>
                                </svg>
                                <p class="text-xl font-medium mb-2">${escapeHtml(data.message || '데이터 로드 실패')}</p>
                                <p class="text-sm opacity-75">검색 조건을 확인해주세요</p>
                            </div>
                        </td>
                    </tr>
                `;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            tbody.style.display = '';
            loadingIndicator.classList.add('hidden');
            tbody.innerHTML = `
                <tr>
                    <td colspan="12" class="px-4 py-12 text-center text-red-400">
                        <svg class="w-12 h-12 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                        <p class="text-lg font-medium">오류가 발생했습니다: ${escapeHtml(error.message)}</p>
                    </td>
                </tr>
            `;
        })
        .finally(() => {
            isLoading = false;
        });
}

// 테이블 렌더링
function renderWithdrawals(withdrawals) {
    const tbody = document.getElementById('withdrawalTableBody');
    withdrawalData = withdrawals;
    
    if (!withdrawals || withdrawals.length === 0) {
        tbody.innerHTML = `
            <tr>
                <td colspan="12" class="px-4 py-12 text-center">
                    <div class="text-gray-400">
                        <svg class="w-16 h-16 mx-auto mb-4 opacity-30" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                        </svg>
                        <p class="text-xl font-medium mb-2">검색 결과가 없습니다</p>
                        <p class="text-sm opacity-75">다른 조건으로 검색해보세요</p>
                    </div>
                </td>
            </tr>
        `;
        return;
    }
    
    tbody.innerHTML = withdrawals.map(withdrawal => {
        let statusClass = '';
        let statusText = '';
        
        switch (withdrawal.status) {
            case 'requested':
                statusClass = 'bg-yellow-500/20 text-yellow-400 border border-yellow-500/30';
                statusText = '신청';
                break;
            case 'pending':
                statusClass = 'bg-orange-500/20 text-orange-400 border border-orange-500/30';
                statusText = '대기';
                break;
            case 'completed':
                statusClass = 'bg-green-500/20 text-green-400 border border-green-500/30';
                statusText = '완료';
                break;
            case 'cancelled':
                statusClass = 'bg-red-500/20 text-red-400 border border-red-500/30';
                statusText = '취소';
                break;
            default:
                statusClass = 'bg-gray-500/20 text-gray-400 border border-gray-500/30';
                statusText = withdrawal.status;
        }
        
        // 계좌정보 포맷
        const accountInfo = withdrawal.bank && withdrawal.account_number 
            ? `<div class="text-xs"><div class="font-medium">${escapeHtml(withdrawal.bank)}</div><div class="text-gray-400">${escapeHtml(withdrawal.account_number)}</div><div class="text-gray-300">${escapeHtml(withdrawal.account_holder || '')}</div></div>`
            : '<span class="text-gray-500">-</span>';

        // 잔액 정보 포맷 (완료된 건만 표시)
        const balanceInfo = withdrawal.status === 'completed' && withdrawal.balance_before
            ? `<div class="text-xs">
                <div class="text-gray-400">처리 전: ${Number(withdrawal.balance_before).toLocaleString()}원</div>
                ${withdrawal.balance_after ? `<div class="text-gray-300">처리 후: ${Number(withdrawal.balance_after).toLocaleString()}원</div>` : ''}
            </div>`
            : '<span class="text-gray-500">-</span>';
        
        // 금액 계산
        const requestedAmount = Number(withdrawal.amount);
        const actualAmount = requestedAmount - WITHDRAWAL_FEE;
        
        // 액션 버튼
        let actionButtons = '';
        if (withdrawal.status === 'requested' || withdrawal.status === 'pending') {
            actionButtons = `
                <div class="flex flex-wrap gap-1">
                    ${withdrawal.status === 'requested' ? `
                        <button onclick="processWithdrawal(${withdrawal.id}, 'pending')" 
                            class="px-2 py-1 text-xs bg-orange-600 hover:bg-orange-700 text-white rounded transition-all duration-300">
                            대기
                        </button>
                    ` : ''}
                    <button onclick="processWithdrawal(${withdrawal.id}, 'completed')" 
                        class="px-2 py-1 text-xs bg-green-600 hover:bg-green-700 text-white rounded transition-all duration-300">
                        완료
                    </button>
                    <button onclick="processWithdrawal(${withdrawal.id}, 'cancelled')" 
                        class="px-2 py-1 text-xs bg-red-600 hover:bg-red-700 text-white rounded transition-all duration-300">
                        취소
                    </button>
                </div>
            `;
        } else {
            actionButtons = '<span class="text-gray-500">-</span>';
        }
        
        return `
            <tr class="hover:bg-gray-700/20 transition-all duration-200">
                <td class="px-4 py-4 font-medium">${escapeHtml(withdrawal.id)}</td>
                <td class="px-4 py-4 font-medium text-white">${escapeHtml(withdrawal.user_name || '-')}</td>
                <td class="px-4 py-4 text-gray-300">${escapeHtml(withdrawal.member_user_id || '-')}</td>
                <td class="px-4 py-4 text-right font-medium">${requestedAmount.toLocaleString()}원</td>
                <td class="px-4 py-4 text-right text-gray-400">${WITHDRAWAL_FEE.toLocaleString()}원</td>
                <td class="px-4 py-4 text-right font-medium text-blue-400">${actualAmount.toLocaleString()}원</td>
                <td class="px-4 py-4 text-gray-300">${escapeHtml(withdrawal.store_code || '-')}</td>
                <td class="px-4 py-4">${accountInfo}</td>
                <td class="px-4 py-4">${balanceInfo}</td>
                <td class="px-4 py-4">
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium ${statusClass}">
                        ${statusText}
                    </span>
                </td>
                <td class="px-4 py-4 text-gray-400 text-xs">${formatDateTime(withdrawal.created_at)}</td>
                <td class="px-4 py-4 text-gray-400 text-xs">${formatDateTime(withdrawal.completed_at) || '-'}</td>
                <td class="px-4 py-4 text-center">${actionButtons}</td>
            </tr>
        `;
    }).join('');
}

// 대시보드 업데이트
function updateDashboard(withdrawals) {
    const today = new Date().toDateString();
    
    let todayCount = 0;
    let todayAmount = 0;
    let pendingCount = 0;
    let pendingAmount = 0;
    let completedCount = 0;
    let completedAmount = 0;
    let totalFees = 0;
    
    withdrawals.forEach(w => {
        const createdDate = new Date(w.created_at).toDateString();
        const isToday = createdDate === today;
        
        if (isToday) {
            todayCount++;
            todayAmount += Number(w.amount || 0);
        }
        
        if (w.status === 'requested' || w.status === 'pending') {
            pendingCount++;
            pendingAmount += Number(w.amount || 0);
        }
        
        if (w.status === 'completed' && isToday) {
            completedCount++;
            completedAmount += Number(w.amount || 0) - WITHDRAWAL_FEE;
            totalFees += WITHDRAWAL_FEE;
        }
    });
    
    document.getElementById('todayCount').textContent = todayCount + '건';
    document.getElementById('todayAmount').textContent = todayAmount.toLocaleString() + '원';
    document.getElementById('pendingCount').textContent = pendingCount + '건';
    document.getElementById('pendingAmount').textContent = pendingAmount.toLocaleString() + '원';
    document.getElementById('completedCount').textContent = completedCount + '건';
    document.getElementById('completedAmount').textContent = completedAmount.toLocaleString() + '원';
    document.getElementById('totalFees').textContent = totalFees.toLocaleString() + '원';
}

// 페이지네이션 렌더링
function renderPagination(totalPages, currentPageNum) {
    const pagination = document.getElementById('pagination');
    
    if (totalPages <= 1) {
        pagination.innerHTML = '';
        return;
    }
    
    let html = '<div class="flex items-center justify-center space-x-2">';
    
    // 이전 버튼
    if (currentPageNum > 1) {
        html += `
            <button onclick="changePage(${currentPageNum - 1})" 
                class="flex items-center px-3 py-2 text-sm text-gray-400 hover:text-white hover:bg-gray-700 rounded-lg transition-all duration-300">
                <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                </svg>
                이전
            </button>
        `;
    }
    
    // 페이지 번호
    for (let i = Math.max(1, currentPageNum - 2); i <= Math.min(totalPages, currentPageNum + 2); i++) {
        html += `
            <button onclick="changePage(${i})" 
                class="px-3 py-2 text-sm rounded-lg transition-all duration-300 ${
                    i === currentPageNum 
                        ? 'bg-gradient-to-r from-blue-600 to-purple-600 text-white font-medium' 
                        : 'text-gray-400 hover:text-white hover:bg-gray-700'
                }">
                ${i}
            </button>
        `;
    }
    
    // 다음 버튼
    if (currentPageNum < totalPages) {
        html += `
            <button onclick="changePage(${currentPageNum + 1})" 
                class="flex items-center px-3 py-2 text-sm text-gray-400 hover:text-white hover:bg-gray-700 rounded-lg transition-all duration-300">
                다음
                <svg class="w-4 h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </button>
        `;
    }
    
    html += '</div>';
    pagination.innerHTML = html;
}

// 페이지 변경
function changePage(page) {
    currentPage = page;
    fetchWithdrawals();
}

// 출금 처리 함수
function processWithdrawal(id, action) {
    let actionText = '';
    let confirmMessage = '';
    
    switch(action) {
        case 'pending':
            actionText = '대기 처리';
            confirmMessage = '이 출금 신청을 대기 상태로 변경하시겠습니까?';
            break;
        case 'completed':
            actionText = '완료 처리';
            confirmMessage = '이 출금 신청을 완료 처리하시겠습니까?\n실제 송금이 완료되었는지 확인하세요.';
            break;
        case 'cancelled':
            actionText = '취소';
            confirmMessage = '이 출금 신청을 취소하시겠습니까?\n이 작업은 되돌릴 수 없습니다.';
            break;
        default:
            showAlert('error', '잘못된 액션입니다.');
            return;
    }
    
    if (!confirm(confirmMessage)) {
        return;
    }
    
    const button = event.target;
    const originalText = button.textContent;
    button.disabled = true;
    button.textContent = '처리중...';
    
    fetch('/ajax/process_withdrawal.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            id: id,
            action: action
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('success', `출금 신청이 ${actionText}되었습니다.`);
            fetchWithdrawals(); // 목록 새로고침
        } else {
            showAlert('error', data.message || `${actionText} 중 오류가 발생했습니다.`);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('error', '처리 중 오류가 발생했습니다.');
    })
    .finally(() => {
        if (button) {
            button.disabled = false;
            button.textContent = originalText;
        }
    });
}

// 날짜시간 포맷팅
function formatDateTime(dateTimeString) {
    if (!dateTimeString) return '';
    const date = new Date(dateTimeString);
    return date.toLocaleDateString('ko-KR') + ' ' + date.toLocaleTimeString('ko-KR', { hour: '2-digit', minute: '2-digit' });
}

// 알림 표시
function showAlert(type, message) {
    // 기존 알림 제거
    const existingAlert = document.querySelector('.alert-message');
    if (existingAlert) {
        existingAlert.remove();
    }
    
    const alertClass = type === 'success' 
        ? 'bg-gradient-to-r from-green-600 to-emerald-600 border-green-500' 
        : 'bg-gradient-to-r from-red-600 to-rose-600 border-red-500';
    
    const alertIcon = type === 'success'
        ? '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>'
        : '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
    
    const alert = document.createElement('div');
    alert.className = `alert-message fixed top-4 right-4 ${alertClass} text-white px-6 py-4 rounded-lg shadow-2xl z-[60] border flex items-center animate-slide-in`;
    alert.innerHTML = `
        <div class="flex items-center">
            ${alertIcon}
            <span class="ml-3 font-medium">${message}</span>
        </div>
    `;
    
    document.body.appendChild(alert);
    
    setTimeout(() => {
        alert.classList.add('animate-fade-out');
        setTimeout(() => alert.remove(), 300);
    }, 3000);
}

// 모바일 사이드바 토글
function toggleSidebar() {
    const sidebar = document.querySelector('.sidebar');
    if (sidebar) {
        sidebar.classList.toggle('hidden');
    }
}

// 엑셀 다운로드 함수
function downloadExcel() {
    const formData = new FormData(document.getElementById('searchForm'));
    const params = new URLSearchParams();
    
    // 검색 조건 파라미터 추가
    for (let [key, value] of formData.entries()) {
        if (value) params.append(key, value);
    }
    
    // 기본 날짜 설정
    const startDate = formData.get('start_date');
    const endDate = formData.get('end_date');
    const defaults = getDefaultDates();
    
    if (!startDate && !endDate) {
        params.append('start_date', defaults.startDate);
        params.append('end_date', defaults.endDate);
    }
    
    // 정렬 파라미터 추가
    params.append('sort', currentSort.column);
    params.append('order', currentSort.direction);
    
    // 엑셀 다운로드 요청임을 표시
    params.append('export', 'excel');
    
    // 다운로드 알림 표시
    showAlert('info', '엑셀 파일을 생성 중입니다...');
    
    // 새 창에서 다운로드 처리
    const downloadUrl = `/ajax/export_withdrawals_excel.php?${params.toString()}`;
    
    // iframe을 이용한 다운로드 (페이지 이동 없이)
    const iframe = document.createElement('iframe');
    iframe.style.display = 'none';
    iframe.src = downloadUrl;
    document.body.appendChild(iframe);
    
    // iframe 로드 완료 후 제거
    iframe.onload = function() {
        setTimeout(() => {
            document.body.removeChild(iframe);
            showAlert('success', '엑셀 파일 다운로드가 시작되었습니다.');
        }, 1000);
    };
    
    // 에러 처리
    iframe.onerror = function() {
        document.body.removeChild(iframe);
        showAlert('error', '엑셀 다운로드 중 오류가 발생했습니다.');
    };
}
</script>

<style>
@keyframes slide-in {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes fade-out {
    from {
        opacity: 1;
    }
    to {
        opacity: 0;
    }
}

.animate-slide-in {
    animation: slide-in 0.3s ease-out;
}

.animate-fade-out {
    animation: fade-out 0.3s ease-out;
}

/* 드롭다운 스크롤바 스타일 */
#store_dropdown::-webkit-scrollbar {
    width: 6px;
}

#store_dropdown::-webkit-scrollbar-track {
    background: #1f2937;
    border-radius: 3px;
}

#store_dropdown::-webkit-scrollbar-thumb {
    background: #4b5563;
    border-radius: 3px;
}

#store_dropdown::-webkit-scrollbar-thumb:hover {
    background: #6b7280;
}

/* 커스텀 스크롤바 */
.overflow-x-auto::-webkit-scrollbar {
    height: 8px;
}

.overflow-x-auto::-webkit-scrollbar-track {
    background: rgba(55, 65, 81, 0.3);
    border-radius: 4px;
}

.overflow-x-auto::-webkit-scrollbar-thumb {
    background: rgba(107, 114, 128, 0.5);
    border-radius: 4px;
}

.overflow-x-auto::-webkit-scrollbar-thumb:hover {
    background: rgba(107, 114, 128, 0.7);
}

/* 정렬 화살표 스타일 */
.sort-indicator {
    display: inline-block;
    margin-left: 4px;
    font-size: 0.75rem;
    color: #9ca3af;
    transition: all 0.2s;
}

th:hover .sort-indicator {
    color: #d1d5db;
}
</style>

</body>
</html>